# Provider Architecture - Scalable Multi-Provider Support

## Overview

The payment provider system is designed to be **highly scalable** and support multiple providers per user. The architecture uses a **Factory Pattern** with a **Provider Registry** to enable easy addition of new providers without modifying existing code.

## Architecture Components

### 1. Provider Factory (`app/services/providers/provider_factory.py`)

**Purpose**: Central registry and factory for all payment providers

**Features**:
- **Registry Pattern**: Maintains dictionaries of available providers
- **Dynamic Provider Selection**: Get provider instances by name
- **Easy Extension**: Add new providers by registering them
- **Default Fallback**: Returns default provider if none specified

**Usage**:
```python
# Get payout provider
provider = ProviderFactory.get_payout_provider("rupeeflow")

# Get payin provider  
provider = ProviderFactory.get_payin_provider("razorpay")

# List available providers
providers = ProviderFactory.list_payout_providers()
```

### 2. Provider Selection Priority

The system uses a **3-tier priority** for provider selection:

1. **Explicit Parameter**: `provider_name` passed to service method
2. **User Configuration**: Stored in `user_services.notes` as JSON
3. **Default Provider**: Falls back to "rupeeflow"

### 3. User-Level Provider Configuration

**Storage**: `user_services.notes` field (JSON format)

**Format**:
```json
{
  "provider": "rupeeflow",
  "other_config": "value"
}
```

**How it works**:
- Each user can have different provider preferences per service
- Stored in the `notes` field of the `user_services` table
- Services automatically check this configuration when no explicit provider is specified

## Adding New Providers

### Step 1: Create Provider Class

```python
# app/services/providers/razorpay_provider.py
from app.services.providers.base_provider import BasePayoutProvider

class RazorpayProvider(BasePayoutProvider):
    def create_payout(self, ...):
        # Implementation
        pass
```

### Step 2: Register Provider

```python
# Option 1: Register in provider_factory.py
_payout_providers = {
    "rupeeflow": RupeeflowProvider,
    "razorpay": RazorpayProvider,  # Add here
}

# Option 2: Dynamic registration
ProviderFactory.register_payout_provider("razorpay", RazorpayProvider)
```

### Step 3: That's it!

The provider is now available for use. No changes needed to services or routes.

## Service Layer Integration

### Payin Service

```python
# Automatically uses user's preferred provider or default
payin_txn = payin_service.initiate_payin(
    user_id=user_id,
    amount=amount,
    # ... other params
    provider_name="razorpay"  # Optional: override user preference
)
```

### Payout Service

```python
# Automatically uses user's preferred provider or default
payout_txn = payout_service.initiate_payout(
    user_id=user_id,
    amount=amount,
    # ... other params
    provider_name="razorpay"  # Optional: override user preference
)
```

## Benefits of This Architecture

### ✅ Scalability
- **Easy to add providers**: Just create class and register
- **No code changes**: Services automatically support new providers
- **Backward compatible**: Existing code continues to work

### ✅ Flexibility
- **Per-user configuration**: Each user can use different providers
- **Per-transaction override**: Can specify provider per transaction
- **Default fallback**: Always has a working default

### ✅ Maintainability
- **Single responsibility**: Each provider is self-contained
- **Clear separation**: Factory handles provider management
- **Testable**: Easy to mock providers for testing

### ✅ Extensibility
- **Future-proof**: Ready for provider-specific features
- **Configuration support**: Can add provider-specific configs
- **Multi-provider support**: Users can have multiple providers active

## Example: User with Multiple Providers

```python
# User A prefers Razorpay for payin
user_service_payin.notes = '{"provider": "razorpay"}'

# User A prefers RupeeFlow for payout  
user_service_payout.notes = '{"provider": "rupeeflow"}'

# User B uses default (RupeeFlow) for both
# No configuration needed
```

## Future Enhancements

1. **Provider-specific configuration**: Store API keys per user/provider
2. **Provider health checks**: Monitor provider availability
3. **Automatic failover**: Switch to backup provider on failure
4. **Provider comparison**: Compare rates/features
5. **A/B testing**: Test different providers per user segment

## Migration Path

**Current State**: Hardcoded RupeeFlow provider
**New State**: Factory-based with user preferences

**Migration**:
- Existing code continues to work (defaults to RupeeFlow)
- Gradually migrate users to preferred providers
- No breaking changes required

