"""add missing ownership_type, pan_number, aadhaar_number columns to kycs table

Revision ID: f63e03e06025
Revises: f7e8d9a0b1c2
Create Date: 2025-01-21 16:05:00.000000

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'f63e03e06025'
down_revision: Union[str, None] = 'f7e8d9a0b1c2'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    # Add ownership_type column if it doesn't exist
    op.execute("""
        DO $$ 
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM information_schema.columns 
                WHERE table_name = 'kycs' AND column_name = 'ownership_type'
            ) THEN
                ALTER TABLE kycs ADD COLUMN ownership_type VARCHAR(50) NOT NULL DEFAULT 'individual';
                CREATE INDEX ix_kycs_ownership_type ON kycs(ownership_type);
            END IF;
        END $$;
    """)
    
    # Remove default after adding (we don't want default in model)
    op.execute("ALTER TABLE kycs ALTER COLUMN ownership_type DROP DEFAULT;")
    
    # Add pan_number column if it doesn't exist
    op.execute("""
        DO $$ 
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM information_schema.columns 
                WHERE table_name = 'kycs' AND column_name = 'pan_number'
            ) THEN
                ALTER TABLE kycs ADD COLUMN pan_number VARCHAR(10);
                CREATE INDEX ix_kycs_pan_number ON kycs(pan_number);
            END IF;
        END $$;
    """)
    
    # Add aadhaar_number column if it doesn't exist
    op.execute("""
        DO $$ 
        BEGIN
            IF NOT EXISTS (
                SELECT 1 FROM information_schema.columns 
                WHERE table_name = 'kycs' AND column_name = 'aadhaar_number'
            ) THEN
                ALTER TABLE kycs ADD COLUMN aadhaar_number VARCHAR(12);
                CREATE INDEX ix_kycs_aadhaar_number ON kycs(aadhaar_number);
            END IF;
        END $$;
    """)
    
    # Ensure document_type and document_number are nullable (they are deprecated)
    op.alter_column('kycs', 'document_type', nullable=True, existing_type=sa.String(100))
    op.alter_column('kycs', 'document_number', nullable=True, existing_type=sa.String(255))
    
    # Drop unique constraint on document_number if it exists
    op.execute("""
        DO $$ 
        BEGIN
            IF EXISTS (
                SELECT 1 FROM pg_constraint 
                WHERE conname = 'kycs_document_number_key'
            ) THEN
                ALTER TABLE kycs DROP CONSTRAINT kycs_document_number_key;
            END IF;
            IF EXISTS (
                SELECT 1 FROM pg_indexes 
                WHERE indexname = 'ix_kycs_document_number' AND indexdef LIKE '%UNIQUE%'
            ) THEN
                DROP INDEX IF EXISTS ix_kycs_document_number;
            END IF;
        END $$;
    """)
    
    # Recreate non-unique index on document_number
    op.create_index('ix_kycs_document_number', 'kycs', ['document_number'], unique=False)


def downgrade() -> None:
    # Drop indexes
    op.drop_index('ix_kycs_document_number', table_name='kycs')
    op.drop_index(op.f('ix_kycs_aadhaar_number'), table_name='kycs')
    op.drop_index(op.f('ix_kycs_pan_number'), table_name='kycs')
    op.drop_index(op.f('ix_kycs_ownership_type'), table_name='kycs')
    
    # Drop columns
    op.drop_column('kycs', 'aadhaar_number')
    op.drop_column('kycs', 'pan_number')
    op.drop_column('kycs', 'ownership_type')
    
    # Restore constraints
    op.alter_column('kycs', 'document_type', nullable=False, existing_type=sa.String(100))
    op.alter_column('kycs', 'document_number', nullable=False, existing_type=sa.String(255))
    op.create_unique_constraint('kycs_document_number_key', 'kycs', ['document_number'])

