"""replace first_name and last_name with company_name in kycs table

Revision ID: f7e8d9a0b1c2
Revises: 80aa6dc0d94a
Create Date: 2025-01-21 16:00:00.000000

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'f7e8d9a0b1c2'
down_revision: Union[str, None] = '80aa6dc0d94a'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    # Add company_name column
    op.add_column('kycs', sa.Column('company_name', sa.String(length=255), nullable=True))
    
    # Migrate data: concatenate first_name and last_name to company_name for existing records
    op.execute("""
        UPDATE kycs 
        SET company_name = TRIM(CONCAT(COALESCE(first_name, ''), ' ', COALESCE(last_name, '')))
        WHERE company_name IS NULL
    """)
    
    # Make company_name NOT NULL after migration
    op.alter_column('kycs', 'company_name', nullable=False)
    
    # Drop old columns
    op.drop_column('kycs', 'first_name')
    op.drop_column('kycs', 'last_name')
    
    # Make document_type and document_number nullable (they are deprecated)
    op.alter_column('kycs', 'document_type', nullable=True)
    
    # Drop unique constraint on document_number if it exists, then make nullable
    # Check if constraint exists before dropping
    op.execute("""
        DO $$ 
        BEGIN
            IF EXISTS (
                SELECT 1 FROM pg_constraint 
                WHERE conname = 'kycs_document_number_key'
            ) THEN
                ALTER TABLE kycs DROP CONSTRAINT kycs_document_number_key;
            END IF;
        END $$;
    """)
    op.alter_column('kycs', 'document_number', nullable=True)


def downgrade() -> None:
    # Add back first_name and last_name columns
    op.add_column('kycs', sa.Column('first_name', sa.String(length=255), nullable=True))
    op.add_column('kycs', sa.Column('last_name', sa.String(length=255), nullable=True))
    
    # Split company_name back (simple split on first space)
    op.execute("""
        UPDATE kycs 
        SET first_name = SPLIT_PART(company_name, ' ', 1),
            last_name = SUBSTRING(company_name FROM LENGTH(SPLIT_PART(company_name, ' ', 1)) + 2)
        WHERE company_name IS NOT NULL
    """)
    
    # Make first_name and last_name NOT NULL
    op.alter_column('kycs', 'first_name', nullable=False)
    op.alter_column('kycs', 'last_name', nullable=False)
    
    # Drop company_name
    op.drop_column('kycs', 'company_name')
    
    # Restore document_type and document_number constraints
    op.alter_column('kycs', 'document_type', nullable=False)
    op.alter_column('kycs', 'document_number', nullable=False)
    op.create_unique_constraint('kycs_document_number_key', 'kycs', ['document_number'])

