"""
Admin API routes for managing all entities
"""

from typing import List, Optional
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from uuid import UUID

from app.core.database import get_db
from app.core.dependencies import get_current_admin, security_scheme_admin
from app.core.exceptions import ValidationError, NotFoundError
from app.models.admin import Admin

# User schemas
from app.schemas.user import UserCreate, UserUpdate, UserResponse
from app.schemas.admin import AdminCreate, AdminUpdate, AdminResponse, AdminLogin
from app.schemas.kyc import KYCCreate, KYCResponse, KYCUpdate
from app.schemas.kyc_doc import KYCDocCreate, KYCDocUpdate, KYCDocResponse
from app.schemas.service import ServiceCreate, ServiceUpdate, ServiceResponse
from app.schemas.user_service import UserServiceCreate, UserServiceUpdate, UserServiceResponse, UserServiceWithServiceInfoResponse
from app.schemas.wallet import WalletResponse
from app.schemas.wallet_transaction import (
    WalletTransactionResponse, 
    WalletDebitRequest,
    WalletCreditRequest
)
from app.common.schemas import PaginatedResponse
from app.schemas.payout_transaction import PayoutTransactionWithUserResponse, PayoutStatusUpdateRequest, PayoutStatsResponse
from app.schemas.payin_transaction import PayinTransactionWithUserResponse, PayinStatusUpdateRequest, UnsettledBalanceResponse, PayinStatsResponse
from app.schemas.charge import ChargeCreate, ChargeUpdate, ChargeResponse, ChargeWithServiceResponse
from app.schemas.auth import TokenResponse
from app.schemas.webhook_log import WebhookLogWithServiceResponse
from app.common.schemas import MessageResponse, PaginatedResponse

# Services
from app.services.user_service import UserService
from app.services.admin_service import AdminService
from app.services.kyc_service import KYCService
from app.services.kyc_doc_service import KYCDocService
from app.services.service_service import ServiceService
from app.services.user_service_service import UserServiceService
from app.services.wallet_service import WalletService
from app.services.payout_service import PayoutService
from app.services.payin_service import PayinService
from app.services.charge_service import ChargeService
from app.services.webhook_log_service import WebhookLogService
from app.services.webhook_sender_service import WebhookSenderService
from app.repositories.wallet_transaction_repository import WalletTransactionRepository
from app.common.enums import TransactionType
from app.core.cache import get_cache

router = APIRouter(prefix="/admin", tags=["admin"])


# Admin Auth Routes
@router.post("/login", response_model=TokenResponse)
def login_admin(credentials: AdminLogin, db: Session = Depends(get_db)):
    """Login admin"""
    service = AdminService(db)
    admin, access_token = service.login(credentials.email, credentials.password)
    return {"access_token": access_token, "token_type": "bearer"}


@router.get("/me", response_model=AdminResponse)
def get_current_admin_info(current_admin: Admin = Depends(get_current_admin)):
    """Get current admin information"""
    return current_admin


# Admin Management Routes
@router.get("/admins", response_model=List[AdminResponse])
def get_admins(
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all admins"""
    service = AdminService(db)
    admins = service.get_all(skip=skip, limit=limit, db=db)
    return admins


@router.get("/admins/{admin_id}", response_model=AdminResponse)
def get_admin(
    admin_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get admin by ID"""
    service = AdminService(db)
    admin = service.get_by_id(str(admin_id), db)
    if not admin:
        raise HTTPException(status_code=404, detail="Admin not found")
    return admin


@router.post("/admins", response_model=AdminResponse, status_code=201)
def create_admin(
    admin_data: AdminCreate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Create a new admin"""
    service = AdminService(db)
    admin_obj = service.create(admin_data, db)
    return admin_obj


@router.put("/admins/{admin_id}", response_model=AdminResponse)
def update_admin(
    admin_id: UUID,
    admin_data: AdminUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update admin"""
    service = AdminService(db)
    admin_obj = service.update(str(admin_id), admin_data, db)
    return admin_obj


@router.delete("/admins/{admin_id}", response_model=MessageResponse)
def delete_admin(
    admin_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Delete admin (cannot delete yourself or the last admin)"""
    from app.repositories.admin_repository import AdminRepository
    
    admin_repo = AdminRepository(db)
    
    # Check if this is the last admin
    total_admins = admin_repo.count()
    if total_admins <= 1:
        raise HTTPException(
            status_code=400,
            detail="Cannot delete the last admin account"
        )
    
    # Prevent self-deletion
    if str(admin_id) == str(current_admin.id):
        raise HTTPException(
            status_code=400,
            detail="Cannot delete your own account"
        )
    
    # Delete the admin
    service = AdminService(db)
    service.repository.delete(str(admin_id))
    return {"message": "Admin deleted successfully"}


# User Management Routes
@router.get("/users", response_model=List[UserResponse])
def get_users(
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all users"""
    service = UserService(db)
    users = service.get_all(skip=skip, limit=limit, db=db)
    return users


@router.get("/users/{user_id}", response_model=UserResponse)
def get_user(
    user_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get user by ID with KYC status (optimized with efficient query)"""
    from app.models.user import User
    from app.models.kyc import KYC
    from sqlalchemy import select
    
    # Get user first
    user = db.query(User).filter(
        User.id == user_id,
        User.deleted_at.is_(None)
    ).first()
    
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Efficiently get latest KYC (id and status) using indexed query (uses index on user_id and created_at)
    # This is a single optimized query that uses indexes efficiently
    latest_kyc = db.query(KYC.id, KYC.status).filter(
        KYC.user_id == user_id
    ).order_by(KYC.created_at.desc()).limit(1).first()
    
    kyc_id = latest_kyc[0] if latest_kyc else None
    kyc_status = latest_kyc[1] if latest_kyc else None
    
    # Build response with KYC id and status
    return {
        "id": user.id,
        "email": user.email,
        "full_name": user.full_name,
        "phone": user.phone,
        "is_active": user.is_active,
        "kyc_id": kyc_id,
        "kyc_status": kyc_status,
        "created_at": user.created_at,
        "updated_at": user.updated_at,
    }


@router.post("/users", response_model=UserResponse, status_code=201)
def create_user(
    user_data: UserCreate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Create a new user"""
    service = UserService(db)
    user_obj = service.create(user_data, db)
    return user_obj


@router.put("/users/{user_id}", response_model=UserResponse)
def update_user(
    user_id: UUID,
    user_data: UserUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update user"""
    service = UserService(db)
    user_obj = service.update(str(user_id), user_data, db)
    return user_obj


@router.delete("/users/{user_id}", response_model=MessageResponse)
def delete_user(
    user_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Delete user (soft delete)"""
    service = UserService(db)
    service.repository.delete(str(user_id))
    return {"message": "User deleted successfully"}


# KYC Management Routes
@router.get("/kycs", response_model=List[KYCResponse])
def get_kycs(
    user_id: Optional[UUID] = None,
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all KYCs with optional user_id filter"""
    service = KYCService(db)
    filters = {}
    if user_id:
        filters["user_id"] = user_id
    kycs = service.get_all(skip=skip, limit=limit, db=db, filters=filters if filters else None)
    return kycs


@router.get("/kycs/{kyc_id}", response_model=KYCResponse)
def get_kyc(
    kyc_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get KYC by ID"""
    service = KYCService(db)
    kyc = service.get_by_id(str(kyc_id), db)
    if not kyc:
        raise HTTPException(status_code=404, detail="KYC not found")
    return kyc


@router.put("/kycs/{kyc_id}/status", response_model=KYCResponse)
def update_kyc_status(
    kyc_id: UUID,
    status: str,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update KYC status"""
    service = KYCService(db)
    kyc = service.update_status(str(kyc_id), status, db)
    return kyc


@router.put("/kycs/{kyc_id}", response_model=KYCResponse)
def update_kyc(
    kyc_id: UUID,
    kyc_data: KYCUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update KYC"""
    service = KYCService(db)
    kyc = service.update(str(kyc_id), kyc_data, db)
    return kyc


@router.get("/kycs/{kyc_id}/documents", response_model=List[KYCDocResponse])
def get_kyc_documents(
    kyc_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all KYC documents for a specific KYC"""
    # Verify KYC exists
    kyc_service = KYCService(db)
    kyc = kyc_service.get_by_id(str(kyc_id), db)
    if not kyc:
        raise HTTPException(status_code=404, detail="KYC not found")
    
    # Get documents
    doc_service = KYCDocService(db)
    docs = doc_service.get_by_kyc_id(kyc_id, db)
    return docs


# KYC Document Management Routes
@router.get("/kyc-docs", response_model=List[KYCDocResponse])
def get_kyc_docs(
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all KYC documents"""
    service = KYCDocService(db)
    docs = service.get_all(skip=skip, limit=limit, db=db)
    return docs


@router.get("/kyc-docs/{doc_id}", response_model=KYCDocResponse)
def get_kyc_doc(
    doc_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get KYC document by ID"""
    service = KYCDocService(db)
    doc = service.get_by_id(str(doc_id), db)
    if not doc:
        raise HTTPException(status_code=404, detail="KYC document not found")
    return doc


@router.put("/kyc-docs/{doc_id}", response_model=KYCDocResponse)
def update_kyc_doc(
    doc_id: UUID,
    doc_data: KYCDocUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update KYC document"""
    service = KYCDocService(db)
    doc = service.update(str(doc_id), doc_data, db)
    return doc


@router.delete("/kyc-docs/{doc_id}", response_model=MessageResponse)
def delete_kyc_doc(
    doc_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Delete KYC document"""
    service = KYCDocService(db)
    service.repository.delete(str(doc_id))
    return {"message": "KYC document deleted successfully"}


# Service Management Routes
@router.get("/services", response_model=List[ServiceResponse])
def get_services(
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all services"""
    service = ServiceService(db)
    services = service.get_all(skip=skip, limit=limit, db=db)
    return services


@router.get("/services/{service_id}", response_model=ServiceResponse)
def get_service(
    service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get service by ID"""
    service = ServiceService(db)
    service_obj = service.get_by_id(str(service_id), db)
    if not service_obj:
        raise HTTPException(status_code=404, detail="Service not found")
    return service_obj


@router.post("/services", response_model=ServiceResponse, status_code=201)
def create_service(
    service_data: ServiceCreate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Create a new service"""
    service = ServiceService(db)
    service_obj = service.create(service_data, db)
    return service_obj


@router.put("/services/{service_id}", response_model=ServiceResponse)
def update_service(
    service_id: UUID,
    service_data: ServiceUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update service"""
    service = ServiceService(db)
    service_obj = service.update(str(service_id), service_data, db)
    return service_obj


@router.put("/services/{service_id}/activate", response_model=ServiceResponse)
def activate_service(
    service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Activate a service"""
    from app.common.enums import ServiceStatus
    service = ServiceService(db)
    service_obj = service.change_status(str(service_id), ServiceStatus.ACTIVE.value, db)
    return service_obj


@router.put("/services/{service_id}/deactivate", response_model=ServiceResponse)
def deactivate_service(
    service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Deactivate a service"""
    from app.common.enums import ServiceStatus
    service = ServiceService(db)
    service_obj = service.change_status(str(service_id), ServiceStatus.INACTIVE.value, db)
    return service_obj


# User Service Management Routes
@router.get("/user-services", response_model=List[UserServiceResponse])
def get_user_services(
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all user services"""
    service = UserServiceService(db)
    user_services = service.get_all(skip=skip, limit=limit, db=db)
    return user_services


@router.get("/user-services/{user_service_id}", response_model=UserServiceResponse)
def get_user_service(
    user_service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get user service by ID"""
    service = UserServiceService(db)
    user_service = service.get_by_id(str(user_service_id), db)
    if not user_service:
        raise HTTPException(status_code=404, detail="User service not found")
    return user_service


@router.get("/users/{user_id}/services", response_model=List[UserServiceWithServiceInfoResponse])
def get_user_services(
    user_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all services for a particular user with their status, including service name and code"""
    from app.repositories.user_service_repository import UserServiceRepository
    from app.repositories.service_repository import ServiceRepository
    
    # Verify user exists
    user_service_instance = UserService(db)
    user = user_service_instance.get_by_id(str(user_id), db)
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    
    # Get user services
    user_service_repo = UserServiceRepository(db)
    user_services = user_service_repo.get_by_user_id(str(user_id))
    
    # Get service details for each user service
    service_repo = ServiceRepository(db)
    result = []
    for user_service in user_services:
        service = service_repo.get_by_id(str(user_service.service_id))
        if service:
            result.append({
                "id": user_service.id,
                "user_id": user_service.user_id,
                "service_id": user_service.service_id,
                "service_name": service.name,
                "service_code": service.code,
                "service_status": service.status,
                "status": user_service.status,
                "notes": user_service.notes,
                "created_at": user_service.created_at,
                "updated_at": user_service.updated_at,
            })
    
    # Sort by service_code ASC
    result.sort(key=lambda x: x["service_code"])
    
    return result


@router.post("/users/{user_id}/services", response_model=UserServiceResponse, status_code=201)
def create_user_service(
    user_id: UUID,
    user_service_data: UserServiceCreate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Create a user service assignment"""
    service = UserServiceService(db)
    user_service = service.create(user_service_data, db, user_id)
    return user_service


@router.put("/user-services/{user_service_id}", response_model=UserServiceResponse)
def update_user_service(
    user_service_id: UUID,
    user_service_data: UserServiceUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update user service"""
    service = UserServiceService(db)
    user_service = service.update(str(user_service_id), user_service_data, db)
    return user_service


@router.put("/user-services/{user_service_id}/activate", response_model=UserServiceResponse)
def activate_user_service(
    user_service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Activate a user service"""
    from app.common.enums import UserServiceStatus
    service = UserServiceService(db)
    user_service = service.change_status(str(user_service_id), UserServiceStatus.ACTIVE.value, db)
    return user_service


@router.put("/user-services/{user_service_id}/deactivate", response_model=UserServiceResponse)
def deactivate_user_service(
    user_service_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Deactivate a user service"""
    from app.common.enums import UserServiceStatus
    service = UserServiceService(db)
    user_service = service.change_status(str(user_service_id), UserServiceStatus.INACTIVE.value, db)
    return user_service


# Wallet Management Routes
@router.get("/wallet", response_model=WalletResponse)
def get_user_wallet(
    user_id: UUID,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get wallet balance for a user"""
    service = WalletService(db)
    wallet = service.get_or_create_wallet(user_id, db)
    return wallet


@router.get("/wallet/transactions", response_model=PaginatedResponse[WalletTransactionResponse])
def get_user_wallet_transactions(
    user_id: Optional[UUID] = None,
    page: Optional[int] = None,
    page_size: Optional[int] = None,
    skip: Optional[int] = None,
    limit: Optional[int] = None,
    type: Optional[str] = None,  # debit or credit
    txn_type: Optional[str] = None,  # transaction type/reason code
    refunded: Optional[bool] = None,  # filter by refunded status
    is_refund: Optional[bool] = None,  # filter by is_refund status
    txnid: Optional[str] = None,  # filter by transaction ID
    username: Optional[str] = None,  # filter by username (email)
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get wallet transaction history with optional filters and pagination
    
    All query parameters are optional:
    - user_id: User ID (if provided, filters by user)
    - page: Page number (default: 1) - use with page_size
    - page_size: Items per page (default: 20, max: 50) - use with page
    - skip: Number of records to skip - use with limit
    - limit: Maximum number of records (default: 20, max: 50) - use with skip
    - type: debit or credit
    - txn_type: transaction type/reason code
    - refunded: true/false
    - is_refund: true/false
    - txnid: transaction ID
    - username: Username (email) to search by
    """
    from app.models.user import User
    from app.models.wallet_transaction import WalletTransaction
    from math import ceil
    
    # Handle pagination: support both page/page_size and skip/limit
    if skip is not None and limit is not None:
        # Use skip/limit directly
        skip_val = skip if skip >= 0 else 0
        limit_val = limit if limit > 0 else 20
        if limit_val > 50:
            limit_val = 50
        # Calculate page for response
        page = (skip_val // limit_val) + 1 if limit_val > 0 else 1
        page_size = limit_val
    else:
        # Use page/page_size (default behavior)
        if page_size is None:
            page_size = 20
        if page is None:
            page = 1
        # Enforce max limit of 50
        if page_size > 50:
            page_size = 50
        if page_size < 1:
            page_size = 20
        if page < 1:
            page = 1
        skip_val = (page - 1) * page_size
    
    service = WalletService(db)
    transactions = service.get_transactions(
        user_id=user_id, 
        db=db, 
        skip=skip_val, 
        limit=page_size,
        username=username,
        type=type,
        txn_type=txn_type,
        refunded=refunded,
        is_refund=is_refund,
        txnid=txnid
    )
    
    # Get total count for pagination (apply same filters)
    base_query = db.query(WalletTransaction)
    
    # Apply filters for count
    if user_id:
        base_query = base_query.filter(WalletTransaction.user_id == user_id)
    if username:
        base_query = base_query.join(User, WalletTransaction.user_id == User.id).filter(
            User.email.ilike(f"%{username}%")
        )
    if type:
        base_query = base_query.filter(WalletTransaction.type == type)
    if txn_type:
        base_query = base_query.filter(WalletTransaction.txn_type == txn_type)
    if refunded is not None:
        base_query = base_query.filter(WalletTransaction.refunded == refunded)
    if is_refund is not None:
        base_query = base_query.filter(WalletTransaction.is_refund == is_refund)
    if txnid:
        base_query = base_query.filter(WalletTransaction.txnid == txnid)
    
    total = base_query.count()
    
    # Get user information for each transaction
    user_ids = {str(t.user_id) for t in transactions}
    users = {}
    if user_ids:
        user_objs = db.query(User).filter(User.id.in_(user_ids)).all()
        users = {str(u.id): u for u in user_objs}
    
    # Build response with user information
    result = []
    for txn in transactions:
        user = users.get(str(txn.user_id))
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "wallet_id": txn.wallet_id,
            "txn_type": txn.txn_type,
            "type": txn.type,
            "amount": txn.amount,
            "opening": txn.opening,
            "closing": txn.closing,
            "txnid": txn.txnid,
            "refunded": txn.refunded,
            "is_refund": txn.is_refund,
            "user_name": user.full_name if user else None,
            "username": user.email if user else None,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


@router.post("/wallet/debit", response_model=WalletTransactionResponse, status_code=201)
def debit_user_wallet(
    debit_request: WalletDebitRequest,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Debit (deduct) amount from user's wallet
    Transaction ID is auto-generated, txn_type is set to 'admin_fund_manage', and main wallet is always updated
    """
    import uuid
    from app.common.enums import TransactionType
    
    # Auto-generate transaction ID
    txnid = str(uuid.uuid4())
    
    service = WalletService(db)
    transaction = service.process_transaction(
        user_id=debit_request.user_id,
        txn_type="admin_fund_manage",
        transaction_type=TransactionType.DEBIT.value,
        amount=debit_request.amount,
        txnid=txnid,
        db=db,
        update_main=True  # Always update main wallet
    )
    return transaction


@router.post("/wallet/credit", response_model=WalletTransactionResponse, status_code=201)
def credit_user_wallet(
    credit_request: WalletCreditRequest,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Credit (add) amount to user's wallet
    Transaction ID is auto-generated, txn_type is set to 'admin_fund_manage', and main wallet is always updated
    """
    import uuid
    from app.common.enums import TransactionType
    
    # Auto-generate transaction ID
    txnid = str(uuid.uuid4())
    
    service = WalletService(db)
    transaction = service.process_transaction(
        user_id=credit_request.user_id,
        txn_type="admin_fund_manage",
        transaction_type=TransactionType.CREDIT.value,
        amount=credit_request.amount,
        txnid=txnid,
        db=db,
        update_main=True  # Always update main wallet
    )
    return transaction



@router.get("/payout/transactions", response_model=PaginatedResponse[PayoutTransactionWithUserResponse])
def get_payout_transactions(
    user_id: Optional[UUID] = None,
    page: Optional[int] = None,
    page_size: Optional[int] = None,
    skip: Optional[int] = None,
    limit: Optional[int] = None,
    status: Optional[str] = None,  # Filter by status
    txnid: Optional[str] = None,  # Filter by transaction ID
    reference_id: Optional[str] = None,  # Filter by user reference ID
    username: Optional[str] = None,  # Filter by username (email)
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get payout transaction history with optional filters and pagination
    
    All query parameters are optional:
    - user_id: User ID (if provided, filters by user)
    - page: Page number (default: 1) - alternative to skip
    - page_size: Items per page (default: 20, max: 50) - alternative to limit
    - skip: Number of records to skip (alternative to page)
    - limit: Maximum number of records (alternative to page_size, max: 50)
    - status: Transaction status (success, failed, pending, error)
    - txnid: Transaction ID
    - reference_id: User reference ID
    - username: Username (email) to search by
    """
    from app.models.user import User
    from app.models.payout_transaction import PayoutTransaction
    from math import ceil
    
    # Handle pagination: support both page/page_size and skip/limit
    if skip is not None and limit is not None:
        # Use skip/limit directly
        skip_val = skip if skip >= 0 else 0
        limit_val = limit if limit > 0 else 20
        if limit_val > 50:
            limit_val = 50
        # Calculate page for response
        page = (skip_val // limit_val) + 1 if limit_val > 0 else 1
        page_size = limit_val
    else:
        # Use page/page_size (default behavior)
        if page_size is None:
            page_size = 20
        if page is None:
            page = 1
        # Enforce max limit of 50
        if page_size > 50:
            page_size = 50
        if page_size < 1:
            page_size = 20
        if page < 1:
            page = 1
        skip_val = (page - 1) * page_size
    
    service = PayoutService(db)
    transactions = service.get_transactions(
        user_id=user_id,
        db=db,
        skip=skip_val,
        limit=page_size,
        username=username,
        status=status,
        txnid=txnid,
        user_reference_id=reference_id
    )
    
    # Get total count for pagination
    base_query = db.query(PayoutTransaction)
    
    # Apply filters for count
    if user_id:
        base_query = base_query.filter(PayoutTransaction.user_id == user_id)
    if username:
        base_query = base_query.join(User, PayoutTransaction.user_id == User.id).filter(
            User.email.ilike(f"%{username}%")
        )
    if status:
        base_query = base_query.filter(PayoutTransaction.status.ilike(f"%{status}%"))
    if txnid:
        base_query = base_query.filter(PayoutTransaction.txnid == txnid)
    if reference_id:
        base_query = base_query.filter(PayoutTransaction.user_reference_id == reference_id)
    
    total = base_query.count()
    
    # Get user information for each transaction
    user_ids = {str(t.user_id) for t in transactions}
    users = {}
    if user_ids:
        user_objs = db.query(User).filter(User.id.in_(user_ids)).all()
        users = {str(u.id): u for u in user_objs}
    
    # Build response with user information
    result = []
    for txn in transactions:
        user = users.get(str(txn.user_id))
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "txnid": txn.txnid,
            "user_reference_id": txn.user_reference_id,
            "amount": txn.amount,
            "charge": txn.charge,
            "bene_name": txn.bene_name,
            "bene_ifsc": txn.bene_ifsc,
            "bene_acc_no": txn.bene_acc_no,
            "status": txn.status,
            "api_response": txn.api_response,
            "api_provider": txn.api_provider,
            "provider_reference_id": txn.provider_reference_id,
            "refunded": txn.refunded,
            "user_name": user.full_name if user else None,
            "username": user.email if user else None,
            "rrn": txn.rrn,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


@router.put("/payout/update-status", response_model=PayoutTransactionWithUserResponse)
def update_payout_status(
    update_data: PayoutStatusUpdateRequest,
    txnid: Optional[str] = None,
    user_reference_id: Optional[str] = None,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Admin endpoint to update payout transaction status
    
    This endpoint allows admin to:
    1. Update the status of a payout transaction (only if current status is "pending")
    2. Optionally refund the wallet if status is failed or error (only if amount was deducted and not already refunded)
    3. Optionally send webhook notification
    
    Parameters:
    - txnid: Transaction ID (either txnid or user_reference_id must be provided)
    - user_reference_id: User reference ID (either txnid or user_reference_id must be provided)
    - update_data: Update request containing status, refund flag, and send_webhook flag
    
    Business Logic:
    - Status can only be updated if current status is "pending"
    - If status is "failed" or "error" and refund is True:
      * Check if wallet transaction with same txnid exists (amount was deducted)
      * Check if not already refunded
      * Process refund using the same txnid (refunds amount + charge)
      * Mark payout transaction as refunded
    - If send_webhook is True:
      * Send webhook notification to user's configured webhook URL
    - All wallet transactions (debit and refund) use the same txnid for tracking
    """
    from app.models.user import User
    from app.repositories.payout_transaction_repository import PayoutTransactionRepository
    
    # Validate that at least one identifier is provided
    if not txnid and not user_reference_id:
        raise HTTPException(
            status_code=400,
            detail="Either 'txnid' or 'user_reference_id' must be provided"
        )
    
    # Validate status
    valid_statuses = ["success", "failed", "pending", "error"]
    if update_data.status.lower() not in valid_statuses:
        raise HTTPException(
            status_code=400,
            detail=f"Invalid status. Must be one of: {', '.join(valid_statuses)}"
        )
    
    # Get payout transaction
    payout_repo = PayoutTransactionRepository(db)
    if txnid:
        payout_txn = payout_repo.get_by_txnid(txnid)
        if not payout_txn:
            raise HTTPException(
                status_code=404,
                detail=f"Payout transaction with txnid '{txnid}' not found"
            )
    else:
        payout_txn = payout_repo.get_by_user_reference_id(user_reference_id)
        if not payout_txn:
            raise HTTPException(
                status_code=404,
                detail=f"Payout transaction with user_reference_id '{user_reference_id}' not found"
            )
    
    # Check if current status is "pending" (only allow update if pending)
    if payout_txn.status.lower() != "pending":
        raise HTTPException(
            status_code=400,
            detail=f"Cannot update status. Current status is '{payout_txn.status}', only 'pending' transactions can be updated"
        )
    
    # Check if transaction is already refunded
    if payout_txn.refunded:
        raise HTTPException(
            status_code=400,
            detail="Transaction is already refunded"
        )
    
    # Get wallet transaction repository to check if amount was deducted
    wallet_txn_repo = WalletTransactionRepository(db)
    
    # Check if wallet transaction exists (amount was deducted)
    # Look for debit transaction with same txnid for payout
    from app.models.wallet_transaction import WalletTransaction
    wallet_debit_txn = db.query(WalletTransaction).filter(
        WalletTransaction.txnid == payout_txn.txnid,
        WalletTransaction.user_id == payout_txn.user_id,
        WalletTransaction.txn_type == "payout",
        WalletTransaction.type == TransactionType.DEBIT.value,
        WalletTransaction.is_refund == False
    ).first()
    
    amount_deducted = wallet_debit_txn is not None
    
    # Process refund if status is failed/error and refund flag is True
    refund_processed = False
    if update_data.status.lower() in ["failed", "error"] and update_data.refund:
        # Check if amount was deducted
        if not amount_deducted:
            raise HTTPException(
                status_code=400,
                detail="Cannot refund: No wallet debit transaction found. Amount was not deducted."
            )
        
        # Check if already refunded (check for refund transaction with same txnid)
        existing_refund = db.query(WalletTransaction).filter(
            WalletTransaction.txnid == payout_txn.txnid,
            WalletTransaction.user_id == payout_txn.user_id,
            WalletTransaction.txn_type == "payout_refund",
            WalletTransaction.is_refund == True
        ).first()
        
        if existing_refund:
            raise HTTPException(
                status_code=400,
                detail="Transaction is already refunded"
            )
        
        # Process refund using the same txnid
        try:
            payout_service = PayoutService(db)
            # Calculate total amount to refund (amount + charge)
            total_refund = payout_txn.amount + payout_txn.charge
            
            # Process refund - uses same txnid for tracking
            refund_wallet_txn = payout_service.wallet_service.process_refund(
                user_id=payout_txn.user_id,
                original_txnid=payout_txn.txnid,  # Original debit transaction txnid
                txn_type="payout_refund",
                amount=total_refund,
                refund_txnid=payout_txn.txnid,  # Use same txnid for refund (as per requirement)
                db=db,
                update_main=True
            )
            refund_processed = True
        except Exception as e:
            # Log error and fail the update
            import logging
            logger = logging.getLogger(__name__)
            logger.error(f"Failed to process refund for payout txnid {payout_txn.txnid}: {str(e)}")
            raise HTTPException(
                status_code=500,
                detail=f"Failed to process refund: {str(e)}"
            )
    
    # Update payout transaction status
    updated_payout = payout_repo.update(
        str(payout_txn.id),
        status=update_data.status.lower()
    )
    
    # Mark as refunded if refund was processed
    if refund_processed:
        updated_payout = payout_repo.update(
            str(payout_txn.id),
            refunded=True
        )
    
    # Send webhook if requested
    webhook_sent = False
    webhook_result = None
    if update_data.send_webhook:
        try:
            webhook_service = WebhookSenderService(db)
            webhook_result = webhook_service.send_webhook(
                service_code="payout",
                txnid=payout_txn.txnid,
                db=db
            )
            webhook_sent = webhook_result.get('success', False)
        except Exception as e:
            # Log error but don't fail the status update
            import logging
            logger = logging.getLogger(__name__)
            logger.error(f"Failed to send webhook for payout txnid {payout_txn.txnid}: {str(e)}")
            # Continue without failing - webhook is optional
    
    # Get user information for response
    user = db.query(User).filter(User.id == payout_txn.user_id).first()
    
    # Build response
    response_data = {
        "id": updated_payout.id,
        "user_id": updated_payout.user_id,
        "txnid": updated_payout.txnid,
        "user_reference_id": updated_payout.user_reference_id,
        "amount": updated_payout.amount,
        "charge": updated_payout.charge,
        "bene_name": updated_payout.bene_name,
        "bene_ifsc": updated_payout.bene_ifsc,
        "bene_acc_no": updated_payout.bene_acc_no,
        "status": updated_payout.status,
        "api_response": updated_payout.api_response,
        "api_provider": updated_payout.api_provider,
        "provider_reference_id": updated_payout.provider_reference_id,
        "refunded": updated_payout.refunded,
        "user_name": user.full_name if user else None,
        "username": user.email if user else None,
        "created_at": updated_payout.created_at,
        "updated_at": updated_payout.updated_at,
    }
    
    # Invalidate payout stats cache when status is updated
    from app.core.cache import get_cache
    cache = get_cache()
    cache.delete_pattern("payout_stats:")
    
    return response_data


@router.get("/payout/stats", response_model=PayoutStatsResponse)
def get_payout_stats(
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get payout statistics for admin dashboard
    
    Returns:
    - Today's statistics:
      * Total charges by status (success, failed, pending, error)
      * Total amount by status (success, failed, pending, error)
      * Total transactions by status (success, failed, pending, error)
    - Current month statistics:
      * Total charges by status
      * Total amount by status
      * Total transactions by status
    - Total till today statistics:
      * Total charges by status
      * Total amount by status
      * Total transactions by status
    """
    from app.models.payout_transaction import PayoutTransaction
    from sqlalchemy import func, case
    from datetime import datetime, date, timedelta
    from decimal import Decimal
    
    cache = get_cache()
    today_str = date.today().isoformat()
    month_str = date.today().strftime("%Y-%m")
    
    # Cache keys for different time periods
    cache_key_today = f"payout_stats:today:{today_str}"
    cache_key_month = f"payout_stats:month:{month_str}"
    cache_key_total = "payout_stats:total"
    
    # Try to get from cache
    cached_today = cache.get(cache_key_today)
    cached_month = cache.get(cache_key_month)
    cached_total = cache.get(cache_key_total)
    
    # If all cached, return immediately
    if cached_today and cached_month and cached_total:
        return PayoutStatsResponse(
            today=cached_today,
            current_month=cached_month,
            total_till_today=cached_total
        )
    
    # Get today's date (start of day)
    today_start = datetime.combine(date.today(), datetime.min.time())
    
    # Get current month start (first day of current month)
    current_month_start = datetime.combine(date.today().replace(day=1), datetime.min.time())
    
    def calculate_stats_by_status(query_filter=None):
        """Helper function to calculate charges, amount, and counts by status"""
        base_query = db.query(PayoutTransaction)
        if query_filter is not None:
            base_query = base_query.filter(query_filter)
        
        # Get charges and amount by status
        charges_by_status = base_query.with_entities(
            PayoutTransaction.status,
            func.coalesce(func.sum(PayoutTransaction.charge), Decimal('0')).label('total_charges')
        ).group_by(PayoutTransaction.status).all()
        
        amount_by_status = base_query.with_entities(
            PayoutTransaction.status,
            func.coalesce(func.sum(PayoutTransaction.amount), Decimal('0')).label('total_amount')
        ).group_by(PayoutTransaction.status).all()
        
        # Get transaction counts by status
        counts_by_status = base_query.with_entities(
            PayoutTransaction.status,
            func.count(PayoutTransaction.id).label('count')
        ).group_by(PayoutTransaction.status).all()
        
        # Initialize dictionaries
        charges = {'success': '0', 'failed': '0', 'pending': '0', 'error': '0'}
        amounts = {'success': '0', 'failed': '0', 'pending': '0', 'error': '0'}
        counts = {'success': 0, 'failed': 0, 'pending': 0, 'error': 0}
        
        # Populate charges
        for status, total in charges_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in charges:
                charges[status_lower] = str(total)
        
        # Populate amounts
        for status, total in amount_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in amounts:
                amounts[status_lower] = str(total)
        
        # Populate counts
        for status, count in counts_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in counts:
                counts[status_lower] = count
        
        return {
            'charges_by_status': charges,
            'amount_by_status': amounts,
            'transactions_by_status': counts
        }
    
    # Calculate statistics (only for missing cache entries)
    today_stats = cached_today or calculate_stats_by_status(
        func.date(PayoutTransaction.created_at) == date.today()
    )
    
    month_stats = cached_month or calculate_stats_by_status(
        PayoutTransaction.created_at >= current_month_start
    )
    
    total_stats = cached_total or calculate_stats_by_status()  # No filter for all time
    
    # Cache with different TTLs based on time period
    # Today: 30 seconds (frequent updates)
    # Current month: 6 hours (moderate updates)
    # Total: 12 hours (rarely changes)
    if not cached_today:
        cache.set(cache_key_today, today_stats, ttl_seconds=30)
    if not cached_month:
        cache.set(cache_key_month, month_stats, ttl_seconds=21600)  # 6 hours
    if not cached_total:
        cache.set(cache_key_total, total_stats, ttl_seconds=43200)  # 12 hours
    
    # Build response
    return PayoutStatsResponse(
        today=today_stats,
        current_month=month_stats,
        total_till_today=total_stats
    )


@router.get("/payin/stats", response_model=PayinStatsResponse)
def get_payin_stats(
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get payin statistics for admin dashboard
    
    Returns:
    - Today's statistics:
      * Total charges by status (success, failed, pending, error)
      * Total amount by status (success, failed, pending, error)
      * Total transactions by status (success, failed, pending, error)
    - Current month statistics:
      * Total charges by status
      * Total amount by status
      * Total transactions by status
    - Total till today statistics:
      * Total charges by status
      * Total amount by status
      * Total transactions by status
    """
    from app.models.payin_transaction import PayinTransaction
    from sqlalchemy import func, case
    from datetime import datetime, date, timedelta
    from decimal import Decimal
    
    cache = get_cache()
    today_str = date.today().isoformat()
    month_str = date.today().strftime("%Y-%m")
    
    # Cache keys for different time periods
    cache_key_today = f"payin_stats:today:{today_str}"
    cache_key_month = f"payin_stats:month:{month_str}"
    cache_key_total = "payin_stats:total"
    
    # Try to get from cache
    cached_today = cache.get(cache_key_today)
    cached_month = cache.get(cache_key_month)
    cached_total = cache.get(cache_key_total)
    
    # If all cached, return immediately
    if cached_today and cached_month and cached_total:
        return PayinStatsResponse(
            today=cached_today,
            current_month=cached_month,
            total_till_today=cached_total
        )
    
    # Get today's date (start of day)
    today_start = datetime.combine(date.today(), datetime.min.time())
    
    # Get current month start (first day of current month)
    current_month_start = datetime.combine(date.today().replace(day=1), datetime.min.time())
    
    def calculate_stats_by_status(query_filter=None):
        """Helper function to calculate charges, amount, and counts by status"""
        base_query = db.query(PayinTransaction)
        if query_filter is not None:
            base_query = base_query.filter(query_filter)
        
        # Get charges and amount by status
        charges_by_status = base_query.with_entities(
            PayinTransaction.status,
            func.coalesce(func.sum(PayinTransaction.charge), Decimal('0')).label('total_charges')
        ).group_by(PayinTransaction.status).all()
        
        amount_by_status = base_query.with_entities(
            PayinTransaction.status,
            func.coalesce(func.sum(PayinTransaction.amount), Decimal('0')).label('total_amount')
        ).group_by(PayinTransaction.status).all()
        
        # Get transaction counts by status
        counts_by_status = base_query.with_entities(
            PayinTransaction.status,
            func.count(PayinTransaction.id).label('count')
        ).group_by(PayinTransaction.status).all()
        
        # Initialize dictionaries
        charges = {'success': '0', 'failed': '0', 'pending': '0', 'error': '0'}
        amounts = {'success': '0', 'failed': '0', 'pending': '0', 'error': '0'}
        counts = {'success': 0, 'failed': 0, 'pending': 0, 'error': 0}
        
        # Populate charges
        for status, total in charges_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in charges:
                charges[status_lower] = str(total)
        
        # Populate amounts
        for status, total in amount_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in amounts:
                amounts[status_lower] = str(total)
        
        # Populate counts
        for status, count in counts_by_status:
            status_lower = status.lower() if status else ''
            if status_lower in counts:
                counts[status_lower] = count
        
        return {
            'charges_by_status': charges,
            'amount_by_status': amounts,
            'transactions_by_status': counts
        }
    
    # Calculate statistics (only for missing cache entries)
    today_stats = cached_today or calculate_stats_by_status(
        func.date(PayinTransaction.created_at) == date.today()
    )
    
    month_stats = cached_month or calculate_stats_by_status(
        PayinTransaction.created_at >= current_month_start
    )
    
    total_stats = cached_total or calculate_stats_by_status()  # No filter for all time
    
    # Cache with different TTLs based on time period
    # Today: 30 seconds (frequent updates)
    # Current month: 6 hours (moderate updates)
    # Total: 12 hours (rarely changes)
    if not cached_today:
        cache.set(cache_key_today, today_stats, ttl_seconds=30)
    if not cached_month:
        cache.set(cache_key_month, month_stats, ttl_seconds=21600)  # 6 hours
    if not cached_total:
        cache.set(cache_key_total, total_stats, ttl_seconds=43200)  # 12 hours
    
    # Build response
    return PayinStatsResponse(
        today=today_stats,
        current_month=month_stats,
        total_till_today=total_stats
    )


@router.get("/payin/transactions", response_model=PaginatedResponse[PayinTransactionWithUserResponse])
def get_payin_transactions(
    user_id: Optional[UUID] = None,
    page: int = 1,
    page_size: int = 20,
    status: Optional[str] = None,  # Filter by status
    txnid: Optional[str] = None,  # Filter by transaction ID
    reference_id: Optional[str] = None,  # Filter by user reference ID
    username: Optional[str] = None,  # Filter by username (email)
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get payin transaction history with optional filters and pagination
    
    All query parameters are optional:
    - user_id: User ID (if provided, filters by user)
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    - status: Transaction status (success, failed, pending, error)
    - txnid: Transaction ID
    - reference_id: User reference ID
    - username: Username (email) to search by
    """
    from app.models.user import User
    from app.models.payin_transaction import PayinTransaction
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    service = PayinService(db)
    transactions = service.get_transactions(
        user_id=user_id,
        db=db,
        skip=skip,
        limit=page_size,
        username=username,
        status=status,
        txnid=txnid,
        user_reference_id=reference_id
    )
    
    # Get total count for pagination
    base_query = db.query(PayinTransaction)
    
    # Apply filters for count
    if user_id:
        base_query = base_query.filter(PayinTransaction.user_id == user_id)
    if username:
        base_query = base_query.join(User, PayinTransaction.user_id == User.id).filter(
            User.email.ilike(f"%{username}%")
        )
    if status:
        base_query = base_query.filter(PayinTransaction.status.ilike(f"%{status}%"))
    if txnid:
        base_query = base_query.filter(PayinTransaction.txnid == txnid)
    if reference_id:
        base_query = base_query.filter(PayinTransaction.user_reference_id == reference_id)
    
    total = base_query.count()
    
    # Get user information for each transaction
    user_ids = {str(t.user_id) for t in transactions}
    users = {}
    if user_ids:
        user_objs = db.query(User).filter(User.id.in_(user_ids)).all()
        users = {str(u.id): u for u in user_objs}
    
    # Build response with user information
    result = []
    for txn in transactions:
        user = users.get(str(txn.user_id))
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "txnid": txn.txnid,
            "user_reference_id": txn.user_reference_id,
            "amount": txn.amount,
            "charge": txn.charge,
            "payee_vpa": txn.payee_vpa,
            "payee_name": txn.payee_name,
            "status": txn.status,
            "api_response": txn.api_response,
            "qr_text": txn.qr_text,
            "payment_url": txn.payment_url,
            "api_provider": txn.api_provider,
            "provider_reference_id": txn.provider_reference_id,
            "refunded": txn.refunded,
            "user_name": user.full_name if user else None,
            "username": user.email if user else None,
            "rrn": txn.rrn,
            "settled": txn.settled,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


@router.put("/payin/update-status", response_model=PayinTransactionWithUserResponse)
def update_payin_status(
    update_data: PayinStatusUpdateRequest,
    txnid: Optional[str] = None,
    user_reference_id: Optional[str] = None,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Admin endpoint to update payin transaction status
    
    This endpoint allows admin to:
    1. Update the status of a payin transaction
    2. Optionally credit the wallet if status is success (only if not already credited)
    3. Optionally send webhook notification
    
    Parameters:
    - txnid: Transaction ID (either txnid or user_reference_id must be provided)
    - user_reference_id: User reference ID (either txnid or user_reference_id must be provided)
    - update_data: Update request containing status, credit_amount flag, and send_webhook flag
    
    Business Logic:
    - If status is "success" and credit_amount is True:
      * Check if wallet transaction with same txnid already exists
      * If not already credited, credit the wallet with the payin amount
    - If status is "failed" or other:
      * No wallet credit (even if credit_amount is True)
    - If send_webhook is True:
      * Send webhook notification to user's configured webhook URL
    """
    from app.models.user import User
    from app.repositories.payin_transaction_repository import PayinTransactionRepository
    
    # Validate that at least one identifier is provided
    if not txnid and not user_reference_id:
        raise HTTPException(
            status_code=400,
            detail="Either 'txnid' or 'user_reference_id' must be provided"
        )
    
    # Validate status
    valid_statuses = ["success", "failed", "pending", "error"]
    if update_data.status.lower() not in valid_statuses:
        raise HTTPException(
            status_code=400,
            detail=f"Invalid status. Must be one of: {', '.join(valid_statuses)}"
        )
    
    # Get payin transaction
    payin_repo = PayinTransactionRepository(db)
    if txnid:
        payin_txn = payin_repo.get_by_txnid(txnid)
        if not payin_txn:
            raise HTTPException(
                status_code=404,
                detail=f"Payin transaction with txnid '{txnid}' not found"
            )
    else:
        payin_txn = payin_repo.get_by_user_reference_id(user_reference_id)
        if not payin_txn:
            raise HTTPException(
                status_code=404,
                detail=f"Payin transaction with user_reference_id '{user_reference_id}' not found"
            )
    
    # Check if transaction is already refunded
    if payin_txn.refunded:
        raise HTTPException(
            status_code=400,
            detail="Cannot update status of a refunded transaction"
        )
    
    # Get wallet transaction repository to check if already credited
    wallet_txn_repo = WalletTransactionRepository(db)
    
    # Check if already credited (for success status with credit_amount flag)
    already_credited = False
    if update_data.status.lower() == "success" and update_data.credit_amount:
        # Check if there's already a credit transaction for this payin txnid
        # Query for payin credit transactions with this txnid
        from app.models.wallet_transaction import WalletTransaction
        existing_wallet_txn = db.query(WalletTransaction).filter(
            WalletTransaction.txnid == payin_txn.txnid,
            WalletTransaction.txn_type == "payin",
            WalletTransaction.type == TransactionType.CREDIT.value,
            WalletTransaction.is_refund == False  # Exclude refund transactions
        ).first()
        if existing_wallet_txn:
            already_credited = True
    
    # Credit wallet if status is success, credit_amount is True, and not already credited
    wallet_credited = False
    if update_data.status.lower() == "success" and update_data.credit_amount and not already_credited:
        try:
            payin_service = PayinService(db)
            wallet_txn = payin_service.wallet_service.process_transaction(
                user_id=payin_txn.user_id,
                txn_type="payin",
                transaction_type=TransactionType.CREDIT.value,
                amount=payin_txn.amount,  # Credit the full amount
                txnid=payin_txn.txnid,  # Same txnid for tracking
                db=db,
                update_main=True
            )
            wallet_credited = True
        except Exception as e:
            # Log error but don't fail the status update
            import logging
            logger = logging.getLogger(__name__)
            logger.error(f"Failed to credit wallet for payin txnid {payin_txn.txnid}: {str(e)}")
            raise HTTPException(
                status_code=500,
                detail=f"Failed to credit wallet: {str(e)}"
            )
    
    # Update payin transaction status
    updated_payin = payin_repo.update(
        str(payin_txn.id),
        status=update_data.status.lower()
    )
    
    # Invalidate payin stats cache
    cache = get_cache()
    cache.delete_pattern("payin_stats:")
    
    # Send webhook if requested
    webhook_sent = False
    webhook_result = None
    if update_data.send_webhook:
        try:
            webhook_service = WebhookSenderService(db)
            webhook_result = webhook_service.send_webhook(
                service_code="payin",
                txnid=payin_txn.txnid,
                db=db
            )
            webhook_sent = webhook_result.get('success', False)
        except Exception as e:
            # Log error but don't fail the status update
            import logging
            logger = logging.getLogger(__name__)
            logger.error(f"Failed to send webhook for payin txnid {payin_txn.txnid}: {str(e)}")
            # Continue without failing - webhook is optional
    
    # Get user information for response
    user = db.query(User).filter(User.id == payin_txn.user_id).first()
    
    # Build response
    response_data = {
        "id": updated_payin.id,
        "user_id": updated_payin.user_id,
        "txnid": updated_payin.txnid,
        "user_reference_id": updated_payin.user_reference_id,
        "amount": updated_payin.amount,
        "charge": updated_payin.charge,
        "payee_vpa": updated_payin.payee_vpa,
        "payee_name": updated_payin.payee_name,
        "status": updated_payin.status,
        "api_response": updated_payin.api_response,
        "qr_text": updated_payin.qr_text,
        "payment_url": updated_payin.payment_url,
        "api_provider": updated_payin.api_provider,
        "provider_reference_id": updated_payin.provider_reference_id,
        "refunded": updated_payin.refunded,
        "user_name": user.full_name if user else None,
        "username": user.email if user else None,
        "rrn": updated_payin.rrn,
        "settled": updated_payin.settled,
        "created_at": updated_payin.created_at,
        "updated_at": updated_payin.updated_at,
    }
    
    return response_data


@router.get("/payin/unsettled-balance", response_model=UnsettledBalanceResponse)
def get_unsettled_balance(
    user_id: Optional[UUID] = None,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get unsettled balance - total amount of successful payin transactions that are not settled (wallet not credited)
    
    This endpoint calculates:
    - Total amount of successful payin transactions where settled = False
    - Total count of such transactions
    - Optionally filtered by user_id
    
    Parameters:
    - user_id: Optional user ID to filter by specific user
    
    Returns:
    - total_unsettled_amount: Total amount of unsettled successful payin transactions
    - total_unsettled_count: Total count of unsettled successful payin transactions
    - user_id: User ID if filtered by user (None if all users)
    """
    from app.models.payin_transaction import PayinTransaction
    from sqlalchemy import func
    from decimal import Decimal
    
    # Build query for successful payin transactions that are not settled
    query = db.query(PayinTransaction).filter(
        PayinTransaction.status == "success",
        PayinTransaction.settled == False,
        PayinTransaction.refunded == False  # Exclude refunded transactions
    )
    
    # Apply user filter if provided
    if user_id:
        query = query.filter(PayinTransaction.user_id == user_id)
    
    # Calculate total unsettled amount and count
    result = query.with_entities(
        func.coalesce(func.sum(PayinTransaction.amount), Decimal('0')).label('total_amount'),
        func.count(PayinTransaction.id).label('total_count')
    ).first()
    
    total_unsettled_amount = result.total_amount if result else Decimal('0')
    total_unsettled_count = result.total_count if result else 0
    
    return UnsettledBalanceResponse(
        total_unsettled_amount=total_unsettled_amount,
        total_unsettled_count=total_unsettled_count,
        user_id=user_id
    )


# Charge Management Routes
@router.post("/charges", response_model=ChargeResponse, status_code=201)
def create_charge(
    charge_data: ChargeCreate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Create a new charge slab"""
    service = ChargeService(db)
    try:
        charge = service.create_charge(
            service_id=charge_data.service_id,
            from_amount=charge_data.from_amount,
            to_amount=charge_data.to_amount,
            type=charge_data.type,
            is_percent=charge_data.is_percent,
            amount=charge_data.amount,
            user_id=charge_data.user_id,
            gst=charge_data.gst,
            db=db
        )
        return charge
    except ValidationError as e:
        raise HTTPException(status_code=400, detail=str(e.detail))
    except NotFoundError as e:
        raise HTTPException(status_code=404, detail=str(e.detail))


@router.get("/charges", response_model=List[ChargeWithServiceResponse])
def get_charges(
    service_id: Optional[UUID] = None,
    user_id: Optional[UUID] = None,
    skip: int = 0,
    limit: int = 100,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get all charges with optional filtering"""
    service = ChargeService(db)
    charges = service.get_charges(
        service_id=service_id,
        user_id=user_id,
        skip=skip,
        limit=limit
    )
    
    # Get service information for each charge
    from app.models.service import Service
    service_ids = {str(c.service_id) for c in charges}
    services = {}
    if service_ids:
        service_objs = db.query(Service).filter(Service.id.in_(service_ids)).all()
        services = {str(s.id): s for s in service_objs}
    
    # Build response with service information
    result = []
    for charge in charges:
        svc = services.get(str(charge.service_id))
        result.append({
            "id": charge.id,
            "user_id": charge.user_id,
            "service_id": charge.service_id,
            "service_name": svc.name if svc else None,
            "service_code": svc.code if svc else None,
            "from_amount": charge.from_amount,
            "to_amount": charge.to_amount,
            "type": charge.type,
            "is_percent": charge.is_percent,
            "amount": charge.amount,
            "created_at": charge.created_at,
            "updated_at": charge.updated_at,
        })
    
    return result


@router.get("/charges/{charge_id}", response_model=ChargeWithServiceResponse)
def get_charge(
    charge_id: int,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Get charge by ID"""
    service = ChargeService(db)
    charge = service.charge_repo.get_by_id(str(charge_id))
    if not charge:
        raise HTTPException(status_code=404, detail="Charge not found")
    
    # Get service information
    from app.models.service import Service
    svc = db.query(Service).filter(Service.id == charge.service_id).first()
    
    return {
        "id": charge.id,
        "user_id": charge.user_id,
        "service_id": charge.service_id,
        "service_name": svc.name if svc else None,
        "service_code": svc.code if svc else None,
        "from_amount": charge.from_amount,
        "to_amount": charge.to_amount,
        "type": charge.type,
        "is_percent": charge.is_percent,
        "amount": charge.amount,
        "created_at": charge.created_at,
        "updated_at": charge.updated_at,
    }


@router.put("/charges/{charge_id}", response_model=ChargeResponse)
def update_charge(
    charge_id: int,
    charge_data: ChargeUpdate,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Update a charge slab"""
    service = ChargeService(db)
    try:
        charge = service.update_charge(
            charge_id=charge_id,
            service_id=charge_data.service_id,
            user_id=charge_data.user_id,
            from_amount=charge_data.from_amount,
            to_amount=charge_data.to_amount,
            type=charge_data.type,
            is_percent=charge_data.is_percent,
            amount=charge_data.amount,
            gst=charge_data.gst,
            db=db
        )
        return charge
    except NotFoundError as e:
        raise HTTPException(status_code=404, detail=str(e.detail))
    except ValidationError as e:
        raise HTTPException(status_code=400, detail=str(e.detail))


@router.delete("/charges/{charge_id}", response_model=MessageResponse)
def delete_charge(
    charge_id: int,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """Delete a charge slab"""
    service = ChargeService(db)
    charge = service.charge_repo.get_by_id(str(charge_id))
    if not charge:
        raise HTTPException(status_code=404, detail="Charge not found")
    
    service.charge_repo.delete(str(charge_id))
    return {"message": "Charge deleted successfully"}


# Webhook Log Management Routes
@router.get("/webhook-logs", response_model=PaginatedResponse[WebhookLogWithServiceResponse])
def get_webhook_logs(
    user_id: Optional[UUID] = None,
    service_id: Optional[UUID] = None,
    txnid: Optional[str] = None,
    page: int = 1,
    page_size: int = 20,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
):
    """
    Get webhook logs with optional filters and pagination
    
    All query parameters are optional:
    - user_id: User ID (if provided, filters by user)
    - service_id: Service ID (if provided, filters by service)
    - txnid: Transaction ID (if provided, filters by transaction ID)
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    """
    from app.models.service import Service
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    service = WebhookLogService(db)
    logs = service.get_logs(
        skip=skip,
        limit=page_size,
        user_id=user_id,
        service_id=service_id,
        txnid=txnid,
        db=db
    )
    
    # Get total count for pagination
    total = service.count_logs(
        user_id=user_id,
        service_id=service_id,
        txnid=txnid,
        db=db
    )
    
    # Get service information for each log
    service_ids = {str(log.service_id) for log in logs}
    services = {}
    if service_ids:
        service_objs = db.query(Service).filter(Service.id.in_(service_ids)).all()
        services = {str(s.id): s for s in service_objs}
    
    # Build response with service information
    result = []
    for log in logs:
        svc = services.get(str(log.service_id))
        result.append({
            "id": log.id,
            "user_id": log.user_id,
            "service_id": log.service_id,
            "service_name": svc.name if svc else None,
            "service_code": svc.code if svc else None,
            "txnid": log.txnid,
            "payload": log.payload,
            "response": log.response,
            "created_at": log.created_at,
            "updated_at": log.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )
