"""
Service API routes (for API key authentication)
"""

from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from sqlalchemy.exc import IntegrityError
from uuid import UUID

from app.core.database import get_db
from app.core.dependencies import get_current_user_by_api_key
from app.models.user import User
from app.schemas.payout_transaction import PayoutTransactionCreate, PayoutTransactionResponse, PayoutInitiateResponse, PayoutStatusResponse
from app.schemas.payin_transaction import PayinTransactionCreate, PayinInitiateResponse, PayinStatusResponse
from app.core.exceptions import ValidationError, NotFoundError
from app.services.payout_service import PayoutService
from app.services.payin_service import PayinService
from decimal import Decimal
from app.repositories.service_repository import ServiceRepository
from app.services.charge_service import ChargeService
router = APIRouter(prefix="/services", tags=["services"])


@router.post("/payout", response_model=PayoutInitiateResponse)
def initiate_payout(
    payout_data: PayoutTransactionCreate,
    current_user: User = Depends(get_current_user_by_api_key),
    db: Session = Depends(get_db)
):
    """
    Initiate a payout transaction
    
    This endpoint:
    1. Checks if user has active payout service
    2. Debits wallet (amount + charge)
    3. Creates payout transaction record
    4. Calls Unitpay API to initiate payout
    5. Returns transaction details
    
    Requires API key authentication via X-API-Key header
    
    Response format:
    - status: success, error, failed, pending
    - msg: Response message
    - data: Transaction details (only safe fields)
    """
    try:
        service = PayoutService(db)
        charge_service = ChargeService(db)
        
        # Get payout service to calculate charges
        service_repo = ServiceRepository(db)
        servicByCode = service_repo.get_by_code('payout')
        if not servicByCode:
            raise NotFoundError(f"Service with code payout not found")
        
        # Calculate charge and GST based on service configuration
        base_charge, gst_amount = charge_service.calculate_charge_and_gst(
            service_id=servicByCode.id, 
            transaction_amount=payout_data.amount, 
            user_id=current_user.id, 
            db=db
        )
        
        # Reject request if charge is not configured (charge is 0.00)
        if base_charge == Decimal('0.00'):
            raise ValidationError("Charge configuration not found for payout service. Please configure charges before initiating payout transactions.")
        
        # Total charge = base charge + GST on charge
        total_charge = base_charge + gst_amount
        
        payout_txn = service.initiate_payout(
            user_id=current_user.id,
            amount=payout_data.amount,
            charge=total_charge,
            bene_name=payout_data.bene_name,
            bene_ifsc=payout_data.bene_ifsc,
            bene_acc_no=payout_data.bene_acc_no,
            user_reference_id=payout_data.reference_id,
            provider_name="unitpay",
            db=db
        )
        
        # Map transaction status to response status
        txn_status = payout_txn.status.lower()
        if txn_status == "success":
            response_status = "success"
        elif txn_status == "failed":
            response_status = "failed"
        elif txn_status == "error":
            response_status = "error"
        else:
            response_status = "pending"
        
        # Prepare safe data for third-party users (no sensitive info)
        response_data = {
            "reference_id": payout_txn.user_reference_id,
            "amount": str(payout_txn.amount),
            "charge": str(payout_txn.charge),
            "status": payout_txn.status,
            "txnid": payout_txn.txnid
        }
        
        msg = "Payout initiated successfully" if response_status == "success" else f"Payout status: {payout_txn.status}"
        
        return PayoutInitiateResponse(
            status=response_status,
            msg=msg,
            data=response_data
        )
    
    except ValidationError as e:
        return PayoutInitiateResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except NotFoundError as e:
        return PayoutInitiateResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except IntegrityError as e:
        # Handle database unique constraint violation
        if "user_reference_id" in str(e.orig).lower() or "uq_pyttxns_user_reference_id" in str(e.orig).lower():
            return PayoutInitiateResponse(
                status="error",
                msg=f"Reference ID '{payout_data.reference_id}' already exists. Please use a unique reference ID.",
                data={}
            )
        return PayoutInitiateResponse(
            status="error",
            msg="Database constraint violation. Please check your input.",
            data={}
        )
    except Exception as e:
        return PayoutInitiateResponse(
            status="error",
            msg=f"Failed to initiate payout: {str(e)}",
            data={}
        )


@router.get("/payout/{user_reference_id}", response_model=PayoutStatusResponse)
def get_payout_status(
    user_reference_id: str,
    current_user: User = Depends(get_current_user_by_api_key),
    db: Session = Depends(get_db)
):
    """
    Get payout transaction status by user reference ID
    
    Requires API key authentication via X-API-Key header
    
    Response format:
    - status: success, error, failed, pending
    - msg: Response message
    - data: Transaction details (only safe fields)
    """
    try:
        service = PayoutService(db)
        
        payout_txn = service.get_payout_status(user_reference_id, current_user.id, db)
        
        # Map transaction status to response status
        txn_status = payout_txn.status.lower()
        if txn_status == "success":
            response_status = "success"
        elif txn_status == "failed":
            response_status = "failed"
        elif txn_status == "error":
            response_status = "error"
        else:
            response_status = "pending"
        
        # Prepare safe data for third-party users (no sensitive info)
        response_data = {
            "reference_id": payout_txn.user_reference_id,
            "amount": str(payout_txn.amount),
            "charge": str(payout_txn.charge),
            "status": payout_txn.status,
            "txnid": payout_txn.txnid,
            "utr": payout_txn.rrn,  # UTR (Unique Transaction Reference) - stored in rrn field
            "created_at": payout_txn.created_at.isoformat() if payout_txn.created_at else None,
            "updated_at": payout_txn.updated_at.isoformat() if payout_txn.updated_at else None
        }
        
        msg = f"Transaction status: {payout_txn.status}"
        
        return PayoutStatusResponse(
            status=response_status,
            msg=msg,
            data=response_data
        )
    
    except NotFoundError as e:
        return PayoutStatusResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except Exception as e:
        return PayoutStatusResponse(
            status="error",
            msg=f"Failed to get payout status: {str(e)}",
            data={}
        )


@router.post("/payin", response_model=PayinInitiateResponse)
def initiate_payin(
    payin_data: PayinTransactionCreate,
    current_user: User = Depends(get_current_user_by_api_key),
    db: Session = Depends(get_db)
):
    """
    Initiate a payin transaction (create payment link/QR)
    
    This endpoint:
    1. Checks if user has active payin service
    2. Calls Unitpay API to generate payment link/QR
    3. Creates payin transaction record with status "pending" (NO wallet credit yet)
    4. Returns payment URL and QR text for user to complete payment
    
    Wallet is credited only when webhook confirms payment completion.
    
    Requires API key authentication via X-API-Key header
    
    Response format:
    - status: success, error, failed, pending
    - msg: Response message
    - data: Transaction details including payment_url and qr_text
    """
    try:  
        service = PayinService(db)
        charge_service = ChargeService(db)
        # Calculate charge (for now, fixed at 0 or can be configured)
        # charge = Decimal('0.00')  # TODO: Calculate charge based on service configuration
        service_repo = ServiceRepository(db)
        servicByCode = service_repo.get_by_code('payin')
        if not servicByCode:
            raise NotFoundError(f"Service with code payin not found")

        # Calculate charge and GST based on service configuration
        base_charge, gst_amount = charge_service.calculate_charge_and_gst(
            service_id=servicByCode.id, 
            transaction_amount=payin_data.amount, 
            user_id=current_user.id, 
            db=db
        )
        
        # Reject request if charge is not configured (charge is 0.00)
        if base_charge == Decimal('0.00'):
            raise ValidationError("Charge configuration not found for payin service. Please configure charges before initiating payin transactions.")
        
        # Total charge = base charge + GST on charge
        total_charge = base_charge + gst_amount
        
        payin_txn = service.initiate_payin(
            user_id=current_user.id,
            amount=payin_data.amount,
            charge=total_charge,
            name=payin_data.name,
            email=payin_data.email,
            mobile=payin_data.mobile,
            vpa=payin_data.vpa,
            user_reference_id=payin_data.reference_id,
            provider_name="myt",
            db=db
        )
        
        # Map transaction status to response status
        txn_status = payin_txn.status.lower()
        if txn_status == "success":
            response_status = "success"
        elif txn_status == "failed":
            response_status = "failed"
        elif txn_status == "error":
            response_status = "error"
        else:
            response_status = "pending"
        
        # Prepare safe data for third-party users (include payment_url and qr_text)
        response_data = {
            "reference_id": payin_txn.user_reference_id,
            "amount": str(payin_txn.amount),
            "charge": str(payin_txn.charge),
            "status": payin_txn.status,
            "txnid": payin_txn.txnid,
            "payment_url": payin_txn.payment_url,
            "qr_text": payin_txn.qr_text
        }
        
        msg = "Payment link generated successfully" if response_status == "success" else f"Payin status: {payin_txn.status}"
        
        return PayinInitiateResponse(
            status=response_status,
            msg=msg,
            data=response_data
        )
    
    except ValidationError as e:
        return PayinInitiateResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except NotFoundError as e:
        return PayinInitiateResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except IntegrityError as e:
        # Handle database unique constraint violation
        if "user_reference_id" in str(e.orig).lower() or "uq_payin_transactions_user_reference_id" in str(e.orig).lower():
            return PayinInitiateResponse(
                status="error",
                msg=f"Reference ID '{payin_data.reference_id}' already exists. Please use a unique reference ID.",
                data={}
            )
        return PayinInitiateResponse(
            status="error",
            msg="Database constraint violation. Please check your input.",
            data={}
        )
    except Exception as e:
        return PayinInitiateResponse(
            status="error",
            msg=f"Failed to initiate payin: {str(e)}",
            data={}
        )


@router.get("/payin/{user_reference_id}", response_model=PayinStatusResponse)
def get_payin_status(
    user_reference_id: str,
    current_user: User = Depends(get_current_user_by_api_key),
    db: Session = Depends(get_db)
):
    """
    Get payin transaction status by user reference ID
    
    Requires API key authentication via X-API-Key header
    
    Response format:
    - status: success, error, failed, pending
    - msg: Response message
    - data: Transaction details (only safe fields)
    """
    try:
        service = PayinService(db)
        
        payin_txn = service.get_payin_status(user_reference_id, current_user.id, db)
        
        # Map transaction status to response status
        txn_status = payin_txn.status.lower()
        if txn_status == "success":
            response_status = "success"
        elif txn_status == "failed":
            response_status = "failed"
        elif txn_status == "error":
            response_status = "error"
        else:
            response_status = "pending"
        
        # Prepare safe data for third-party users (no sensitive info)
        response_data = {
            "reference_id": payin_txn.user_reference_id,
            "amount": str(payin_txn.amount),
            "charge": str(payin_txn.charge),
            "status": payin_txn.status,
            "txnid": payin_txn.txnid,
            "utr": payin_txn.rrn,  # UTR (Unique Transaction Reference) - stored in rrn field
            "payment_url": payin_txn.payment_url,
            "qr_text": payin_txn.qr_text,
            "created_at": payin_txn.created_at.isoformat() if payin_txn.created_at else None,
            "updated_at": payin_txn.updated_at.isoformat() if payin_txn.updated_at else None
        }
        
        msg = f"Transaction status: {payin_txn.status}"
        
        return PayinStatusResponse(
            status=response_status,
            msg=msg,
            data=response_data
        )
    
    except NotFoundError as e:
        return PayinStatusResponse(
            status="error",
            msg=str(e),
            data={}
        )
    except Exception as e:
        return PayinStatusResponse(
            status="error",
            msg=f"Failed to get payin status: {str(e)}",
            data={}
        )

