"""
User API routes
"""

from typing import List, Optional
from fastapi import APIRouter, Depends, HTTPException, UploadFile, File, Form, Security
from sqlalchemy.orm import Session
from uuid import UUID
import os
from datetime import datetime

from app.core.database import get_db
from app.core.dependencies import get_current_user, security_scheme_user
from app.models.user import User
from app.schemas.user import UserUpdate, UserResponse, UserLogin
from app.schemas.kyc import (
    KYCCreate, 
    KYCResponse, 
    KYCUpdate, 
    KYCStatusResponse, 
    KYCConfigResponse,
    RequiredDocumentsResponse,
    DocumentStatusInfo
)
from app.schemas.kyc_doc import KYCDocCreate, KYCDocResponse
from app.schemas.service import ServiceResponse
from app.schemas.user_service import UserServiceCreate, UserServiceResponse, UserServiceWithDetailsResponse
from app.schemas.api_key import APIKeyGenerateResponse, APIKeyResponse, APIKeyUpdate, APIKeyAddIPRequest, APIKeyIPsResponse
from app.schemas.wallet import WalletResponse
from app.schemas.wallet_transaction import WalletTransactionResponse
from app.schemas.payout_transaction import PayoutTransactionResponse
from app.schemas.payin_transaction import PayinTransactionResponse, UnsettledBalanceResponse
from app.schemas.auth import TokenResponse
from app.schemas.webhook_url import WebhookURLCreate, WebhookURLResponse
from app.schemas.webhook_log import WebhookLogUserResponse
from app.schemas.webhook_sender import WebhookSendRequest, WebhookSendResponse
from app.common.schemas import PaginatedResponse, MessageResponse
from app.services.user_service import UserService
from app.services.kyc_service import KYCService
from app.services.kyc_doc_service import KYCDocService
from app.services.service_service import ServiceService
from app.services.user_service_service import UserServiceService
from app.services.api_key_service import APIKeyService
from app.services.wallet_service import WalletService
from app.services.payout_service import PayoutService
from app.services.payin_service import PayinService
from app.services.webhook_url_service import WebhookURLService
from app.services.webhook_log_service import WebhookLogService
from app.services.webhook_sender_service import WebhookSenderService
from app.core.exceptions import NotFoundError, ValidationError

router = APIRouter(prefix="/users", tags=["users"])

@router.post("/login", response_model=TokenResponse)
def login(credentials: UserLogin, db: Session = Depends(get_db)):
    """Login user"""
    service = UserService(db)
    user, access_token = service.login(credentials.email, credentials.password)
    return {"access_token": access_token, "token_type": "bearer"}


@router.get("/me", response_model=UserResponse)
def get_current_user_info(current_user: User = Depends(get_current_user)):
    """Get current user information"""
    return current_user


@router.put("/me", response_model=UserResponse)
def update_current_user(
    user_data: UserUpdate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Update current user information"""
    service = UserService(db)
    updated_user = service.update(str(current_user.id), user_data, db)
    return updated_user


# KYC Routes
@router.get("/kyc/config", response_model=KYCConfigResponse)
def get_kyc_config():
    """
    Get KYC configuration including ownership types and required documents
    """
    from app.common.kyc_config import load_kyc_config
    from app.schemas.kyc import OwnershipTypeInfo
    
    config = load_kyc_config()
    
    # Convert ownership types to response format
    ownership_types = {}
    for key, value in config.get("ownership_types", {}).items():
        ownership_types[key] = OwnershipTypeInfo(**value)
    
    return KYCConfigResponse(
        ownership_types=ownership_types,
        document_descriptions=config.get("document_descriptions", {})
    )


@router.get("/kyc/status", response_model=KYCStatusResponse)
def get_kyc_status(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get current KYC status
    Returns: not_submitted, pending, under_review, completed, or rejected
    Note: User can only have one KYC. If rejected, they can resubmit which updates the existing KYC.
    """
    service = KYCService(db)
    status_data = service.get_user_kyc_status(current_user.id, db)
    
    # Convert KYC model to response schema if exists
    if status_data["kyc"]:
        kyc_response = KYCResponse.model_validate(status_data["kyc"])
        return KYCStatusResponse(status=status_data["status"], kyc=kyc_response)
    
    return KYCStatusResponse(status=status_data["status"], kyc=None)


@router.get("/kyc/required-documents", response_model=RequiredDocumentsResponse)
def get_required_documents(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get required documents for user's KYC with submission status
    Returns list of required and optional documents with their status (not_submitted, submitted, pending, approved, rejected)
    Only accessible after user has submitted KYC.
    """
    service = KYCService(db)
    docs_data = service.get_user_required_documents(current_user.id, db)
    
    # Convert to response schemas
    required_docs = [DocumentStatusInfo(**doc) for doc in docs_data["required_documents"]]
    optional_docs = [DocumentStatusInfo(**doc) for doc in docs_data["optional_documents"]]
    
    return RequiredDocumentsResponse(
        ownership_type=docs_data["ownership_type"],
        ownership_type_name=docs_data["ownership_type_name"],
        required_documents=required_docs,
        optional_documents=optional_docs
    )


@router.post("/kyc/submit", response_model=KYCResponse, status_code=201)
def submit_kyc(
    kyc_data: KYCCreate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Submit basic KYC information
    
    This endpoint collects only basic details (ownership type, name, address, PAN, Aadhaar, etc.).
    Documents should be uploaded separately using the document upload API.
    
    Note: User can only have one KYC. If previously rejected, this will update the existing KYC.
    """
    service = KYCService(db)
    kyc = service.create(kyc_data, db, current_user.id)
    return kyc


@router.get("/kyc", response_model=List[KYCResponse])
def get_user_kycs(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Get all KYC records for current user"""
    service = KYCService(db)
    kycs = service.get_all(filters={"user_id": current_user.id}, db=db)
    return kycs


@router.post("/kyc/submit-for-review", response_model=KYCResponse)
def submit_kyc_for_review(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Submit KYC for review
    
    Validates that:
    - KYC details are submitted
    - All required documents are submitted
    - No documents are rejected
    - KYC is not already rejected
    
    After submission, KYC status will be set to 'under_review' and user cannot edit details until admin approves or rejects.
    """
    service = KYCService(db)
    kyc = service.submit_for_review(current_user.id, db)
    return kyc


@router.get("/kyc/{kyc_id}", response_model=KYCResponse)
def get_kyc(
    kyc_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Get KYC by ID"""
    service = KYCService(db)
    kyc = service.get_by_id(str(kyc_id), db)
    if not kyc or kyc.user_id != current_user.id:
        raise HTTPException(status_code=404, detail="KYC not found")
    return kyc


@router.get("/kyc/{kyc_id}/documents", response_model=List[KYCDocResponse])
def get_kyc_documents(
    kyc_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Get all KYC documents for a specific KYC"""
    # Verify KYC exists and belongs to user
    kyc_service = KYCService(db)
    kyc = kyc_service.get_by_id(str(kyc_id), db)
    if not kyc or kyc.user_id != current_user.id:
        raise HTTPException(status_code=404, detail="KYC not found")
    
    # Get documents
    doc_service = KYCDocService(db)
    docs = doc_service.get_by_kyc_id(kyc_id, db)
    return docs


# KYC Document Upload Routes
UPLOAD_DIR = "uploads/kyc_docs"
os.makedirs(UPLOAD_DIR, exist_ok=True)


@router.post("/kyc/upload-document", response_model=KYCDocResponse, status_code=201)
async def upload_kyc_document(
    kyc_id: UUID = Form(...),
    document_type: str = Form(...),
    document_number: str = Form(...),
    front: UploadFile = File(...),
    back: Optional[UploadFile] = File(None),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Upload KYC document
    
    - kyc_id: KYC ID (required)
    - document_type: Type of document (required)
    - document_number: Document number (required)
    - front: Front side file (required)
    - back: Back side file (optional, required for some document types)
    """
    from app.services.kyc_service import KYCService
    from app.core.exceptions import NotFoundError, ConflictError
    from app.common.enums import KYCStatus
    
    # Verify KYC belongs to current user
    kyc_service = KYCService(db)
    kyc = kyc_service.repository.get_by_id(str(kyc_id))
    if not kyc or kyc.user_id != current_user.id:
        raise NotFoundError("KYC not found")
    
    # Check if this is a re-upload of a rejected document
    from app.repositories.kyc_doc_repository import KYCDocRepository
    from app.common.enums import KYCDocStatus
    
    doc_repo = KYCDocRepository(db)
    existing_doc = doc_repo.get_by_user_and_document_type(current_user.id, document_type)
    is_rejected_doc_reupload = existing_doc and existing_doc.status == KYCDocStatus.REJECTED.value
    
    # Prevent document uploads when KYC is under review or approved
    # Exception: Allow re-uploading rejected documents even when KYC is under review
    if kyc.status == KYCStatus.UNDER_REVIEW.value and not is_rejected_doc_reupload:
        raise ConflictError("Cannot upload documents while KYC is under review. You can only re-upload rejected documents.")
    if kyc.status == KYCStatus.APPROVED.value:
        raise ConflictError("Cannot upload documents for approved KYC")
    
    # Compress documents before saving
    from app.services.document_compression_service import DocumentCompressionService
    compression_service = DocumentCompressionService()
    
    # Read and compress front file
    front_contents = await front.read()
    compressed_front, front_mime_type = compression_service.compress_document(
        file_content=front_contents,
        mime_type=front.content_type,
        filename=front.filename
    )
    
    # Save front file
    timestamp = int(datetime.utcnow().timestamp())
    front_filename = f"{current_user.id}_{timestamp}_front_{front.filename}"
    front_filename = front_filename.replace(" ", "_")
    front_file_path = os.path.join(UPLOAD_DIR, front_filename)
    
    with open(front_file_path, "wb") as f:
        f.write(compressed_front)
    
    front_stored_path = f"static/kyc_docs/{front_filename}"
    front_file_name = front.filename
    
    # Save back file if provided
    back_file_path = None
    back_file_name = None
    if back:
        back_contents = await back.read()
        compressed_back, back_mime_type = compression_service.compress_document(
            file_content=back_contents,
            mime_type=back.content_type,
            filename=back.filename
        )
        
        back_filename = f"{current_user.id}_{timestamp}_back_{back.filename}"
        back_filename = back_filename.replace(" ", "_")
        back_file_path_full = os.path.join(UPLOAD_DIR, back_filename)
        
        with open(back_file_path_full, "wb") as f:
            f.write(compressed_back)
        
        back_file_path = f"static/kyc_docs/{back_filename}"
        back_file_name = back.filename
    
    # Get file size and mime type (use compressed size and detected mime type)
    file_size = str(len(compressed_front))
    mime_type = front_mime_type
    
    # Create document record
    doc_data = KYCDocCreate(
        kyc_id=kyc_id,
        document_type=document_type,
        document_number=document_number,
        front_file_path=front_stored_path,
        front_file_name=front_file_name,
        back_file_path=back_file_path,
        back_file_name=back_file_name,
        file_size=file_size,
        mime_type=mime_type
    )
    
    service = KYCDocService(db)
    doc = service.create(doc_data, db, current_user.id)
    return doc


@router.get("/kyc/documents", response_model=List[KYCDocResponse])
def get_user_kyc_documents(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Get all KYC documents for current user"""
    service = KYCDocService(db)
    docs = service.get_all(filters={"user_id": current_user.id}, db=db)
    return docs


# Service Routes (View only for users)
@router.get("/services", response_model=List[ServiceResponse])
def get_services(
    skip: int = 0,
    limit: int = 100,
    db: Session = Depends(get_db)
):
    """Get all available services"""
    service = ServiceService(db)
    services = service.get_all(skip=skip, limit=limit, db=db)
    return services


@router.get("/services/{service_id}", response_model=ServiceResponse)
def get_service(
    service_id: UUID,
    db: Session = Depends(get_db)
):
    """Get service by ID"""
    service = ServiceService(db)
    service_obj = service.get_by_id(str(service_id), db)
    if not service_obj:
        raise HTTPException(status_code=404, detail="Service not found")
    return service_obj


# User Service Routes
@router.post("/my-services", response_model=UserServiceResponse, status_code=201)
def subscribe_to_service(
    service_data: UserServiceCreate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Subscribe to a service"""
    service = UserServiceService(db)
    user_service = service.create(service_data, db, current_user.id)
    return user_service


@router.get("/my-services", response_model=List[UserServiceWithDetailsResponse])
def get_my_services(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get all services for current user
    Returns all services with their status (active/inactive for this user)
    If user has 1 service out of 5 total, returns all 5 with 1 active and 4 inactive
    """
    service = UserServiceService(db)
    services_data = service.get_user_services_with_details(current_user.id, db)
    return services_data


@router.put("/my-services/{service_id}/activate", response_model=UserServiceResponse)
def activate_my_service(
    service_id: UUID,
    notes: Optional[str] = None,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Activate a service for current user
    Creates user_service record if it doesn't exist, otherwise updates status to active
    """
    service = UserServiceService(db)
    user_service = service.activate_service_for_user(current_user.id, service_id, db, notes)
    return user_service


@router.put("/my-services/{service_id}/deactivate", response_model=UserServiceResponse)
def deactivate_my_service(
    service_id: UUID,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Deactivate a service for current user
    Updates status to inactive
    """
    service = UserServiceService(db)
    user_service = service.deactivate_service_for_user(current_user.id, service_id, db)
    return user_service


# API Key Management Routes
@router.post("/api-keys/generate", response_model=APIKeyGenerateResponse, status_code=201)
def generate_api_key(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Generate or regenerate API key pair for current user
    If user already has an API key, it will be updated (not created new)
    Secret is only visible once during generation - save it immediately!
    """
    service = APIKeyService(db)
    result = service.generate_or_regenerate(current_user.id, db)
    return result


@router.get("/api-keys", response_model=APIKeyResponse)
def get_my_api_key(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get current user's API key (without secret)
    Secret is never returned after initial generation
    """
    service = APIKeyService(db)
    api_key = service.get_by_user_id(current_user.id, db)
    
    # Return without secret
    return {
        "id": api_key.id,
        "user_id": api_key.user_id,
        "api_key": api_key.api_key,
        "active": api_key.active,
        "ips": api_key.ips or [],
        "created_at": api_key.created_at,
        "updated_at": api_key.updated_at,
    }


@router.put("/api-keys", response_model=APIKeyResponse)
def update_my_api_key(
    api_key_data: APIKeyUpdate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Update API key settings (active status and IP whitelist)
    """
    service = APIKeyService(db)
    api_key = service.update(current_user.id, api_key_data, db)
    
    # Return without secret
    return {
        "id": api_key.id,
        "user_id": api_key.user_id,
        "api_key": api_key.api_key,
        "active": api_key.active,
        "ips": api_key.ips or [],
        "created_at": api_key.created_at,
        "updated_at": api_key.updated_at,
    }


@router.post("/api-keys/whitelist-ips", response_model=APIKeyResponse)
def add_ip_to_whitelist(
    ip_data: APIKeyAddIPRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Add an IP address to the API key whitelist
    """
    service = APIKeyService(db)
    api_key = service.add_ip_to_whitelist(current_user.id, ip_data.ip, db)
    
    # Return without secret
    return {
        "id": api_key.id,
        "user_id": api_key.user_id,
        "api_key": api_key.api_key,
        "active": api_key.active,
        "ips": api_key.ips or [],
        "created_at": api_key.created_at,
        "updated_at": api_key.updated_at,
    }


@router.get("/api-keys/whitelist-ips", response_model=APIKeyIPsResponse)
def get_whitelisted_ips(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get list of whitelisted IP addresses for current user's API key
    """
    service = APIKeyService(db)
    ips = service.get_whitelisted_ips(current_user.id, db)
    return {"ips": ips}


@router.delete("/api-keys/whitelist-ips/{ip}", response_model=APIKeyResponse)
def remove_ip_from_whitelist(
    ip: str,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Remove an IP address from the API key whitelist
    """
    service = APIKeyService(db)
    api_key = service.remove_ip_from_whitelist(current_user.id, ip, db)
    
    # Return without secret
    return {
        "id": api_key.id,
        "user_id": api_key.user_id,
        "api_key": api_key.api_key,
        "active": api_key.active,
        "ips": api_key.ips or [],
        "created_at": api_key.created_at,
        "updated_at": api_key.updated_at,
    }


# Wallet Management Routes
@router.get("/wallet", response_model=WalletResponse)
def get_my_wallet(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get current user's wallet balance
    """
    service = WalletService(db)
    wallet = service.get_or_create_wallet(current_user.id, db)
    return wallet


@router.get("/wallet/transactions", response_model=PaginatedResponse[WalletTransactionResponse])
def get_my_wallet_transactions(
    page: int = 1,
    page_size: int = 20,
    type: Optional[str] = None,  # debit or credit
    txn_type: Optional[str] = None,  # transaction type/reason code
    refunded: Optional[bool] = None,  # filter by refunded status
    is_refund: Optional[bool] = None,  # filter by is_refund status
    txnid: Optional[str] = None,  # filter by transaction ID
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get wallet transaction history for current user with filters and pagination
    
    Filters:
    - type: debit or credit
    - txn_type: transaction type/reason code
    - refunded: true/false
    - is_refund: true/false
    - txnid: transaction ID
    
    Pagination:
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    """
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    # Build query with filters BEFORE pagination
    from app.models.wallet_transaction import WalletTransaction
    base_query = db.query(WalletTransaction).filter(WalletTransaction.user_id == current_user.id)
    
    # Apply filters to query
    if type:
        base_query = base_query.filter(WalletTransaction.type == type)
    if txn_type:
        base_query = base_query.filter(WalletTransaction.txn_type == txn_type)
    if refunded is not None:
        base_query = base_query.filter(WalletTransaction.refunded == refunded)
    if is_refund is not None:
        base_query = base_query.filter(WalletTransaction.is_refund == is_refund)
    if txnid:
        base_query = base_query.filter(WalletTransaction.txnid == txnid)
    
    # Get total count with filters
    total = base_query.count()
    
    # Apply pagination and get transactions
    transactions = base_query.order_by(WalletTransaction.created_at.desc()).offset(skip).limit(page_size).all()
    
    # Build response with user information
    result = []
    for txn in transactions:
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "wallet_id": txn.wallet_id,
            "txn_type": txn.txn_type,
            "type": txn.type,
            "amount": txn.amount,
            "opening": txn.opening,
            "closing": txn.closing,
            "txnid": txn.txnid,
            "refunded": txn.refunded,
            "is_refund": txn.is_refund,
            "user_name": current_user.full_name,
            "username": current_user.email,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )



@router.get("/payout/transactions", response_model=PaginatedResponse[PayoutTransactionResponse])
def get_my_payout_transactions(
    page: int = 1,
    page_size: int = 20,
    status: Optional[str] = None,  # Filter by status
    txnid: Optional[str] = None,  # Filter by transaction ID
    reference_id: Optional[str] = None,  # Filter by user reference ID
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get payout transaction history for current user with filters and pagination
    
    Filters:
    - status: Transaction status (success, failed, pending, error)
    - txnid: Transaction ID
    - reference_id: User reference ID
    
    Pagination:
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    """
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    service = PayoutService(db)
    transactions = service.get_transactions(
        user_id=current_user.id,
        db=db,
        skip=skip,
        limit=page_size,
        status=status,
        txnid=txnid,
        user_reference_id=reference_id
    )
    
    # Get total count for pagination
    from app.models.payout_transaction import PayoutTransaction
    base_query = db.query(PayoutTransaction).filter(PayoutTransaction.user_id == current_user.id)
    
    # Apply same filters for count
    if status:
        base_query = base_query.filter(PayoutTransaction.status.ilike(f"%{status}%"))
    if txnid:
        base_query = base_query.filter(PayoutTransaction.txnid == txnid)
    if reference_id:
        base_query = base_query.filter(PayoutTransaction.user_reference_id == reference_id)
    
    total = base_query.count()
    
    # Build response without api_response for users
    result = []
    for txn in transactions:
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "txnid": txn.txnid,
            "user_reference_id": txn.user_reference_id,
            "amount": txn.amount,
            "charge": txn.charge,
            "bene_name": txn.bene_name,
            "bene_ifsc": txn.bene_ifsc,
            "bene_acc_no": txn.bene_acc_no,
            "status": txn.status,
            "api_provider": txn.api_provider,
            "provider_reference_id": txn.provider_reference_id,
            "rrn": txn.rrn,
            "refunded": txn.refunded,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


@router.get("/payin/transactions", response_model=PaginatedResponse[PayinTransactionResponse])
def get_my_payin_transactions(
    page: int = 1,
    page_size: int = 20,
    status: Optional[str] = None,  # Filter by status
    txnid: Optional[str] = None,  # Filter by transaction ID
    reference_id: Optional[str] = None,  # Filter by user reference ID
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get payin transaction history for current user with filters and pagination
    
    Filters:
    - status: Transaction status (success, failed, pending, error)
    - txnid: Transaction ID
    - reference_id: User reference ID
    
    Pagination:
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    """
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    service = PayinService(db)
    transactions = service.get_transactions(
        user_id=current_user.id,
        db=db,
        skip=skip,
        limit=page_size,
        status=status,
        txnid=txnid,
        user_reference_id=reference_id
    )
    
    # Get total count for pagination
    from app.models.payin_transaction import PayinTransaction
    base_query = db.query(PayinTransaction).filter(PayinTransaction.user_id == current_user.id)
    
    # Apply same filters for count
    if status:
        base_query = base_query.filter(PayinTransaction.status.ilike(f"%{status}%"))
    if txnid:
        base_query = base_query.filter(PayinTransaction.txnid == txnid)
    if reference_id:
        base_query = base_query.filter(PayinTransaction.user_reference_id == reference_id)
    
    total = base_query.count()
    
    # Build response without api_response for users
    result = []
    for txn in transactions:
        result.append({
            "id": txn.id,
            "user_id": txn.user_id,
            "txnid": txn.txnid,
            "user_reference_id": txn.user_reference_id,
            "amount": txn.amount,
            "charge": txn.charge,
            "payee_vpa": txn.payee_vpa,
            "payee_name": txn.payee_name,
            "status": txn.status,
            "qr_text": txn.qr_text,
            "payment_url": txn.payment_url,
            "api_provider": txn.api_provider,
            "provider_reference_id": txn.provider_reference_id,
            "refunded": txn.refunded,
            "rrn": txn.rrn,
            "settled": txn.settled,
            "created_at": txn.created_at,
            "updated_at": txn.updated_at,
        })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


@router.get("/payin/unsettled-balance", response_model=UnsettledBalanceResponse)
def get_my_unsettled_balance(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get unsettled balance for current user - total amount of successful payin transactions that are not settled (wallet not credited)
    
    This endpoint calculates:
    - Total amount of successful payin transactions where settled = False for the current user
    - Total count of such transactions
    
    Returns:
    - total_unsettled_amount: Total amount of unsettled successful payin transactions
    - total_unsettled_count: Total count of unsettled successful payin transactions
    - user_id: Current user's ID
    """
    from app.models.payin_transaction import PayinTransaction
    from sqlalchemy import func
    from decimal import Decimal
    
    # Build query for successful payin transactions that are not settled for current user
    query = db.query(PayinTransaction).filter(
        PayinTransaction.user_id == current_user.id,
        PayinTransaction.status == "success",
        PayinTransaction.settled == False,
        PayinTransaction.refunded == False  # Exclude refunded transactions
    )
    
    # Calculate total unsettled amount and count
    result = query.with_entities(
        func.coalesce(func.sum(PayinTransaction.amount), Decimal('0')).label('total_amount'),
        func.count(PayinTransaction.id).label('total_count')
    ).first()
    
    total_unsettled_amount = result.total_amount if result else Decimal('0')
    total_unsettled_count = result.total_count if result else 0
    
    return UnsettledBalanceResponse(
        total_unsettled_amount=total_unsettled_amount,
        total_unsettled_count=total_unsettled_count,
        user_id=current_user.id
    )


# Webhook URL Management Routes
@router.post("/webhook-url", response_model=WebhookURLResponse, status_code=201)
def set_webhook_url(
    webhook_data: WebhookURLCreate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Set or update webhook URL for current user
    Only one webhook URL can be set per user. If user already has a webhook URL, it will be updated.
    """
    service = WebhookURLService(db)
    webhook_url = service.set_webhook_url(current_user.id, webhook_data.url, db)
    return webhook_url


@router.get("/webhook-url", response_model=WebhookURLResponse)
def get_webhook_url(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get current user's webhook URL
    """
    service = WebhookURLService(db)
    webhook_url = service.get_by_user_id(current_user.id, db)
    if not webhook_url:
        raise HTTPException(status_code=404, detail="Webhook URL not found")
    return webhook_url


# Webhook Log Routes
@router.get("/webhook-logs", response_model=PaginatedResponse[WebhookLogUserResponse])
def get_my_webhook_logs(
    txnid: Optional[str] = None,
    page: int = 1,
    page_size: int = 20,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Get webhook logs for current user with optional filters and pagination
    
    Filters:
    - txnid: Transaction ID (optional)
    
    Pagination:
    - page: Page number (default: 1)
    - page_size: Items per page (default: 20, max: 50)
    """
    from app.models.service import Service
    from math import ceil
    
    # Enforce max limit of 50
    if page_size > 50:
        page_size = 50
    if page_size < 1:
        page_size = 20
    if page < 1:
        page = 1
    
    skip = (page - 1) * page_size
    
    service = WebhookLogService(db)
    logs = service.get_logs(
        skip=skip,
        limit=page_size,
        user_id=current_user.id,
        txnid=txnid,
        db=db
    )
    
    # Get total count for pagination
    total = service.count_logs(
        user_id=current_user.id,
        txnid=txnid,
        db=db
    )
    
    # Get service information for each log
    service_ids = {str(log.service_id) for log in logs}
    services = {}
    if service_ids:
        service_objs = db.query(Service).filter(Service.id.in_(service_ids)).all()
        services = {str(s.id): s for s in service_objs}
    
    # Build response with service_code (not service_id) for users
    result = []
    for log in logs:
        svc = services.get(str(log.service_id))
        if svc:  # Only include logs with valid service
            result.append({
                "id": log.id,
                "service_code": svc.code,
                "txnid": log.txnid,
                "payload": log.payload,
                "response": log.response,
                "created_at": log.created_at,
                "updated_at": log.updated_at,
            })
    
    total_pages = ceil(total / page_size) if total > 0 else 0
    
    return PaginatedResponse(
        items=result,
        total=total,
        page=page,
        page_size=page_size,
        total_pages=total_pages
    )


# Webhook Sender Routes
@router.post("/webhook/send", response_model=WebhookSendResponse)
def send_webhook_manually(
    webhook_request: WebhookSendRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Manually send webhook for a transaction
    
    This endpoint allows users to manually trigger a webhook for a specific transaction.
    The webhook will be sent to the user's configured webhook URL.
    
    Requirements:
    - User must have a webhook URL configured
    - Transaction must exist and belong to the current user
    - Service code must be valid
    
    Returns:
    - success: Whether the webhook was sent successfully
    - webhook_url: The URL where the webhook was sent
    - message: Human-readable message
    - response: HTTP response from webhook endpoint (if available)
    - error: Error message (if any)
    """
    try:
        webhook_service = WebhookSenderService(db)
        
        # Verify transaction belongs to current user before sending
        from app.repositories.payout_transaction_repository import PayoutTransactionRepository
        from app.repositories.payin_transaction_repository import PayinTransactionRepository
        from app.repositories.wallet_transaction_repository import WalletTransactionRepository
        
        # Check if transaction belongs to user
        transaction_found = False
        payout_repo = PayoutTransactionRepository(db)
        payin_repo = PayinTransactionRepository(db)
        wallet_repo = WalletTransactionRepository(db)
        
        payout_txn = payout_repo.get_by_txnid(webhook_request.txnid)
        if payout_txn and str(payout_txn.user_id) == str(current_user.id):
            transaction_found = True
        
        if not transaction_found:
            payin_txn = payin_repo.get_by_txnid(webhook_request.txnid)
            if payin_txn and str(payin_txn.user_id) == str(current_user.id):
                transaction_found = True
        
        if not transaction_found:
            wallet_txn = wallet_repo.get_by_txnid(webhook_request.txnid)
            if wallet_txn and str(wallet_txn.user_id) == str(current_user.id):
                transaction_found = True
        
        if not transaction_found:
            raise NotFoundError("Transaction not found or does not belong to you")
        
        # Send webhook
        result = webhook_service.send_webhook(
            service_code=webhook_request.service_code,
            txnid=webhook_request.txnid,
            payload=None,  # Auto-generate payload from transaction
            db=db
        )
        
        # Return response
        if result['success']:
            return WebhookSendResponse(
                success=True,
                webhook_url=result.get('webhook_url'),
                message="Webhook sent successfully",
                response=result.get('response'),
                error=None
            )
        else:
            error_msg = result.get('error', 'Unknown error')
            if error_msg == 'User has not configured webhook URL':
                raise HTTPException(
                    status_code=400,
                    detail="You have not configured a webhook URL. Please set your webhook URL first."
                )
            
            return WebhookSendResponse(
                success=False,
                webhook_url=result.get('webhook_url'),
                message="Webhook sending failed",
                response=result.get('response'),
                error=error_msg
            )
    
    except NotFoundError as e:
        raise HTTPException(status_code=404, detail=str(e))
    
    except ValidationError as e:
        raise HTTPException(status_code=400, detail=str(e))
    
    except HTTPException:
        # Re-raise HTTP exceptions as-is
        raise
    
    except Exception as e:
        raise HTTPException(
            status_code=500,
            detail=f"Failed to send webhook: {str(e)}"
        )
