"""
Webhook API routes
"""

import json
import os
from datetime import datetime
from fastapi import APIRouter, Depends, Form, Request
from starlette.requests import Request
from starlette.responses import Response
from sqlalchemy.orm import Session
from pydantic import BaseModel, Field
from typing import Optional
from decimal import Decimal

from app.core.database import get_db
from app.services.payout_service import PayoutService
from app.services.payin_service import PayinService
from app.core.exceptions import NotFoundError
from app.services.webhook_sender_service import WebhookSenderService

import logging
logger = logging.getLogger(__name__)

router = APIRouter(prefix="/webhooks", tags=["webhooks"])

# Logs directory - will be mounted as volume
LOGS_DIR = "/app/logs"
PAYIN_WEBHOOK_LOG_FILE = os.path.join(LOGS_DIR, "rupeeflow_payin_webhooks.log")
PAYOUT_WEBHOOK_LOG_FILE = os.path.join(LOGS_DIR, "rupeeflow_payout_webhooks.log")
UNITPAY_PAYIN_WEBHOOK_LOG_FILE = os.path.join(LOGS_DIR, "unitpay_payin_webhooks.log")
UNITPAY_PAYOUT_WEBHOOK_LOG_FILE = os.path.join(LOGS_DIR, "unitpay_payout_webhooks.log")
MYT_PAYIN_WEBHOOK_LOG_FILE = os.path.join(LOGS_DIR, "myt_payin_webhooks.log")

# Ensure logs directory exists
os.makedirs(LOGS_DIR, exist_ok=True)


def log_webhook_to_file(webhook_type: str, payload: dict, response: dict, error: Optional[str] = None, provider: str = "rupeeflow"):
    """
    Log webhook data to a file
    
    Args:
        webhook_type: Type of webhook ("payin" or "payout")
        payload: Webhook payload received
        response: Response sent back
        error: Error message if any
        provider: Provider name (default: "rupeeflow")
    """
    try:
        # Select log file based on webhook type and provider
        if provider == "unitpay" and webhook_type == "payin":
            log_file = UNITPAY_PAYIN_WEBHOOK_LOG_FILE
            log_type = "unitpay_payin_webhook"
        elif provider == "unitpay" and webhook_type == "payout":
            log_file = UNITPAY_PAYOUT_WEBHOOK_LOG_FILE
            log_type = "unitpay_payout_webhook"
        elif provider == "myt" and webhook_type == "payin":
            log_file = MYT_PAYIN_WEBHOOK_LOG_FILE
            log_type = "myt_payin_webhook"
        elif webhook_type == "payin":
            log_file = PAYIN_WEBHOOK_LOG_FILE
            log_type = "rupeeflow_payin_webhook"
        elif webhook_type == "payout":
            log_file = PAYOUT_WEBHOOK_LOG_FILE
            log_type = "rupeeflow_payout_webhook"
        else:
            log_file = os.path.join(LOGS_DIR, f"{provider}_{webhook_type}_webhooks.log")
            log_type = f"{provider}_{webhook_type}_webhook"
        
        log_entry = {
            "timestamp": datetime.utcnow().isoformat(),
            "type": log_type,
            "payload": payload,
            "response": response,
            "error": error
        }
        
        # Append to log file
        with open(log_file, "a", encoding="utf-8") as f:
            f.write(json.dumps(log_entry, ensure_ascii=False) + "\n")
    except Exception as e:
        # Don't fail webhook if logging fails
        print(f"Failed to log webhook: {str(e)}")


class RupeeflowWebhookPayload(BaseModel):
    """Rupeeflow webhook payload schema"""
    orderId: str = Field(..., description="Rupeeflow order ID")
    utr: str = Field(..., description="UTR number")
    status: str = Field(..., description="Transaction status (SUCCESS, FAILED, PENDING)")
    amount: Optional[float] = Field(None, description="Transaction amount")
    txnRefId: Optional[str] = Field(None, description="Bank transaction reference ID (UTR)")
    payoutId: Optional[str] = Field(None, description="Original payout ID")


@router.post("/rupeeflow/payout")
async def rupeeflow_payout_webhook(
    payload: RupeeflowWebhookPayload,
    db: Session = Depends(get_db)
):
    """
    Webhook endpoint for Rupeeflow payout status updates
    
    This endpoint receives webhook callbacks from Rupeeflow when payout status changes.
    It updates the payout transaction status in the database.
    All webhook data is logged to a file for audit purposes.
    
    Webhook payload structure:
    {
        "orderId": "mpout_01JJ3TQW8PMXXXXXXXXXJTTV08",
        "status": "SUCCESS",
        "amount": 100,
        "txnRefId": "ABC094320545671",
        "payoutId": "payout_123"
    }
    
    Returns HTTP 200 on success (required by Rupeeflow)
    """
    # Prepare payload dict for logging
    payload_dict = payload.model_dump()
    logger.info(f"Rupeeflow payout webhook payload: {payload_dict}")
    try:
        service = PayoutService(db)
        
        # Convert amount to Decimal if provided
        amount_decimal = None
        if payload.amount is not None:
            amount_decimal = Decimal(str(payload.amount))
        
        # Handle webhook
        updated_payout = service.handle_webhook(
            order_id=payload.orderId,
            status=payload.status,
            txn_ref_id=payload.utr,
            payout_id=payload.payoutId,
            amount=amount_decimal,
            db=db
        )
        
        # Prepare success response
        response = {
            "status": "success",
            "message": "Webhook processed successfully",
            "orderId": payload.orderId
        }
        
        # Log webhook to file
        log_webhook_to_file("payout", payload_dict, response)
        #send webhok
        webhook_service = WebhookSenderService(db)
        webhook_result = webhook_service.send_webhook(
            service_code="payout",
            txnid=updated_payout.txnid,
            db=db
        )
        # Return 200 status as required by Rupeeflow
        return response
        
    except NotFoundError as e:
        # Transaction not found - still return 200 to prevent retries
        # Log this for investigation
        error_msg = f"Transaction not found: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": payload.orderId
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payout", payload_dict, response, error=error_msg)
        
        return response
        
    except Exception as e:
        # Return 200 even on error to prevent webhook retries
        # In production, you might want to log this and return 500 for retries
        error_msg = f"Webhook processing failed: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": payload.orderId
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payout", payload_dict, response, error=error_msg)
        
        return response


class RupeeflowPayinWebhookPayload(BaseModel):
    """Rupeeflow payin webhook payload schema"""
    orderId: str = Field(..., description="Rupeeflow order ID")
    utr: str = Field(..., description="UTR number")
    status: str = Field(..., description="Transaction status (SUCCESS, FAILED, PENDING)")
    amount: Optional[float] = Field(None, description="Transaction amount")
    txnRefId: Optional[str] = Field(None, description="Rupeeflow reference number")


@router.post("/rupeeflow/payin") 
async def rupeeflow_payin_webhook(
    payload: RupeeflowPayinWebhookPayload,
    db: Session = Depends(get_db)
):
    """
    Webhook endpoint for Rupeeflow payin status updates
    
    This endpoint receives webhook callbacks from Rupeeflow when payin payment is completed.
    It updates the payin transaction status and credits the wallet when payment is successful.
    All webhook data is logged to a file for audit purposes.
    
    Webhook payload structure:
    {
        "orderId": "sjdbshjdbsjhdsd",
        "status": "SUCCESS",
        "amount": 100
    }
    
    Returns HTTP 200 on success (required by Rupeeflow)
    """
    # Prepare payload dict for logging
    payload_dict = payload.model_dump()
    logger.info(f"Rupeeflow payin webhook payload: {payload_dict}")
    try:
        service = PayinService(db)
        
        # Convert amount to Decimal if provided
        amount_decimal = None
        if payload.amount is not None:
            amount_decimal = Decimal(str(payload.amount))
        
        # Handle webhook (this will credit wallet if status is SUCCESS)
        updated_payin = service.handle_webhook(
            order_id=payload.orderId,
            status=payload.status,
            amount=amount_decimal,
            rrn=payload.txnRefId,
            db=db
        )
        
        # Prepare success response
        response = {
            "status": "success",
            "message": "Webhook processed successfully",
            "orderId": payload.orderId
        }
        
        # Log webhook to file
        log_webhook_to_file("payin", payload_dict, response)
        
        # Return 200 status as required by Rupeeflow
        return response
        
    except NotFoundError as e:
        # Transaction not found - still return 200 to prevent retries
        # Log this for investigation
        error_msg = f"Transaction not found: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": payload.orderId
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payin", payload_dict, response, error=error_msg)
        
        return response
        
    except Exception as e:
        # Return 200 even on error to prevent webhook retries
        # In production, you might want to log this and return 500 for retries
        error_msg = f"Webhook processing failed: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": payload.orderId
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payin", payload_dict, response, error=error_msg)
        
        return response


@router.post("/unitpay/payin")
async def unitpay_payin_webhook(
    amount: str = Form(...),
    apitxnid: Optional[str] = Form(None),
    refid: str = Form(...),
    status: str = Form(...),
    txnid: str = Form(...),
    upiid: Optional[str] = Form(None),
    utr: Optional[str] = Form(None),
    db: Session = Depends(get_db)
):
    """
    Webhook endpoint for Unitpay payin status updates
    
    This endpoint receives webhook callbacks from Unitpay when payin payment is completed.
    It updates the payin transaction status and credits the wallet when payment is successful.
    All webhook data is logged to a file for audit purposes.
    
    Webhook payload structure (Form data):
    - amount: Transaction amount
    - apitxnid: API transaction ID (optional)
    - refid: Reference ID / Order ID (used to find transaction)
    - status: Transaction status (success, failed, pending)
    - txnid: Transaction ID
    - upiid: UPI ID (optional)
    - utr: UTR number (optional)
    
    Returns HTTP 200 on success (required by Unitpay)
    """
    # Prepare payload dict for logging
    payload_dict = {
        "amount": amount,
        "apitxnid": apitxnid,
        "refid": refid,
        "status": status,
        "txnid": txnid,
        "upiid": upiid,
        "utr": utr
    }
    logger.info(f"Unitpay payin webhook payload: {payload_dict}")
    
    try:
        service = PayinService(db)
        
        # Convert amount to Decimal if provided
        amount_decimal = None
        if amount:
            try:
                amount_decimal = Decimal(str(amount))
            except:
                amount_decimal = None
        
        # Map Unitpay status to our status format
        # Unitpay sends "success", "failed", "pending" (lowercase)
        status_upper = status.upper() if status else "PENDING"
        
        # Use refid as order_id (this is the provider_reference_id from Unitpay)
        # refid is the orderid from the API response, stored as provider_reference_id
        order_id = refid
        
        # Handle webhook (this will credit wallet if status is SUCCESS)
        # Search by provider_reference_id (refid) - this is what Unitpay sends as the order ID
        # Use utr as rrn (reference number)
        updated_payin = service.handle_webhook(
            order_id=order_id,
            status=status_upper,
            amount=amount_decimal,
            rrn=utr,
            db=db
        )
        
        # Prepare success response
        response = {
            "status": "success",
            "message": "Webhook processed successfully",
            "orderId": order_id
        }
        
        # Log webhook to file
        log_webhook_to_file("payin", payload_dict, response, provider="unitpay")
        
        # Return 200 status as required by Unitpay
        return response
        
    except NotFoundError as e:
        # Transaction not found - still return 200 to prevent retries
        # Log this for investigation
        error_msg = f"Transaction not found: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": refid
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payin", payload_dict, response, error=error_msg, provider="unitpay")
        
        return response
        
    except Exception as e:
        # Return 200 even on error to prevent webhook retries
        # In production, you might want to log this and return 500 for retries
        error_msg = f"Webhook processing failed: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": refid
        }
        
        # Log webhook to file with error
        log_webhook_to_file("payin", payload_dict, response, error=error_msg, provider="unitpay")
        
        return response


class MytPayinWebhookPayload(BaseModel):
    """MytPiPay payin webhook payload schema"""
    type: str = Field(..., description="Webhook type (payin)")
    status: str = Field(..., description="Transaction status (credit=success, debit=failed)")
    order_id: str = Field(..., description="Order ID / provider_reference_id")
    amount: Optional[float] = Field(None, description="Transaction amount")
    utr: Optional[str] = Field(None, description="UTR number")
    message: Optional[str] = Field(None, description="Status message")


@router.post("/myt/payin")
async def myt_payin_webhook(
    payload: MytPayinWebhookPayload,
    db: Session = Depends(get_db)
):
    """
    Webhook endpoint for MytPiPay payin status updates

    This endpoint receives webhook callbacks from MytPiPay when payin payment is completed.
    It updates the payin transaction status and credits the wallet when status is "credit".

    Webhook payload structure (JSON POST):
    {
        "type": "payin",
        "status": "credit",
        "order_id": "1770393206535604702",
        "amount": 10,
        "utr": "409159900102",
        "message": "UPI Collection SUCCESS"
    }

    Returns HTTP 200 on success (required by MytPiPay)
    """
    payload_dict = payload.model_dump()
    logger.info(f"MytPiPay payin webhook payload: {payload_dict}")
    try:
        service = PayinService(db)

        amount_decimal = None
        if payload.amount is not None:
            amount_decimal = Decimal(str(payload.amount))

        # Map MYT status to our format: "credit" -> SUCCESS, else FAILED
        status_map = {"credit": "SUCCESS", "debit": "FAILED"}
        status_upper = status_map.get(payload.status.lower(), payload.status.upper())

        updated_payin = service.handle_webhook(
            order_id=payload.order_id,
            status=status_upper,
            amount=amount_decimal,
            rrn=payload.utr,
            db=db
        )

        response = {
            "status": "success",
            "message": "Webhook processed successfully",
            "order_id": payload.order_id
        }

        log_webhook_to_file("payin", payload_dict, response, provider="myt")
        return response

    except NotFoundError as e:
        error_msg = f"Transaction not found: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "order_id": payload.order_id
        }
        log_webhook_to_file("payin", payload_dict, response, error=error_msg, provider="myt")
        return response

    except Exception as e:
        error_msg = f"Webhook processing failed: {str(e)}"
        response = {
            "status": "error",
            "message": error_msg,
            "order_id": payload.order_id
        }
        log_webhook_to_file("payin", payload_dict, response, error=error_msg, provider="myt")
        return response


@router.post("/unitpay/payout")
async def unitpay_payout_webhook(
    request: Request,
    db: Session = Depends(get_db)
):
    """
    Webhook endpoint for Unitpay payout status updates
    
    Handles webhook callbacks from Unitpay when payout status changes.
    Accepts both JSON and form data (multipart/form-data or application/x-www-form-urlencoded).
    Updates payout transaction status and sends webhook to customer.
    """
    payload_dict = {}
    refid = ""
    txnid = ""
    
    try:
        # Get all headers
        headers_dict = dict(request.headers)
        content_type = request.headers.get("content-type", "").lower()
        
        # Get raw request body for logging
        raw_body = await request.body()
        raw_body_str = raw_body.decode('utf-8', errors='ignore')
        
        # Log raw request
        logger.info("=" * 80)
        logger.info("[Unitpay Payout Webhook] RAW WEBHOOK DATA RECEIVED")
        logger.info("=" * 80)
        logger.info(f"[Unitpay Payout Webhook] Method: {request.method}")
        logger.info(f"[Unitpay Payout Webhook] URL: {request.url}")
        logger.info(f"[Unitpay Payout Webhook] Path: {request.url.path}")
        logger.info(f"[Unitpay Payout Webhook] Content-Type: {content_type}")
        logger.info(f"[Unitpay Payout Webhook] Raw Body (string): {raw_body_str}")
        logger.info("=" * 80)
        
        # Parse payload based on content type
        # Note: We read raw_body first, so we parse from raw_body_str to avoid consuming the stream twice
        if "application/json" in content_type:
            # Parse JSON from raw body string
            try:
                payload_dict = json.loads(raw_body_str)
                logger.info(f"[Unitpay Payout Webhook] Parsed JSON payload: {payload_dict}")
            except Exception as e:
                logger.error(f"[Unitpay Payout Webhook] Failed to parse JSON: {e}", exc_info=True)
                payload_dict = {}
        elif "multipart/form-data" in content_type:
            # For multipart/form-data, parse manually from raw body
            try:
                import re
                # Extract boundary from content-type
                boundary_match = re.search(r'boundary=([^;\s]+)', content_type)
                if boundary_match:
                    boundary = boundary_match.group(1).strip('"\'')
                    logger.info(f"[Unitpay Payout Webhook] Extracted boundary from content-type: '{boundary}'")
                    
                    # Parse multipart body manually
                    payload_dict = {}
                    # In multipart body, boundaries are prefixed with -- and may have leading dashes
                    # Find the actual boundary marker in the body (it starts with --)
                    # The boundary value itself might already include dashes
                    boundary_marker = f'--{boundary}'
                    logger.info(f"[Unitpay Payout Webhook] Looking for boundary marker: '{boundary_marker}'")
                    
                    # Split by the boundary marker
                    parts = raw_body_str.split(boundary_marker)
                    logger.info(f"[Unitpay Payout Webhook] Split into {len(parts)} parts")
                    
                    for i, part in enumerate(parts):
                        # Skip empty parts and the closing marker (last part with just --)
                        part_stripped = part.strip()
                        if not part_stripped or part_stripped == '--' or (i == len(parts) - 1 and part_stripped.endswith('--')):
                            logger.info(f"[Unitpay Payout Webhook] Skipping part {i}: empty or closing marker")
                            continue
                        
                        # Look for Content-Disposition header
                        if 'Content-Disposition' in part:
                            # Extract field name
                            name_match = re.search(r'name="([^"]+)"', part)
                            if name_match:
                                field_name = name_match.group(1)
                                # Extract field value (everything after \r\n\r\n until next boundary or end)
                                value_match = re.search(r'\r\n\r\n(.+?)(?:\r\n--|$)', part, re.DOTALL)
                                if value_match:
                                    field_value = value_match.group(1).strip('\r\n')
                                    payload_dict[field_name] = field_value
                                    logger.info(f"[Unitpay Payout Webhook] Extracted field: {field_name} = {field_value}")
                                else:
                                    logger.warning(f"[Unitpay Payout Webhook] Could not extract value for field: {field_name}")
                            else:
                                logger.warning(f"[Unitpay Payout Webhook] Could not extract field name from part {i}")
                    
                    logger.info(f"[Unitpay Payout Webhook] Parsed multipart form data payload: {payload_dict}")
                else:
                    logger.warning("[Unitpay Payout Webhook] Could not extract boundary from Content-Type")
                    payload_dict = {}
            except Exception as e:
                logger.error(f"[Unitpay Payout Webhook] Failed to parse multipart form data: {e}", exc_info=True)
                import traceback
                logger.error(f"[Unitpay Payout Webhook] Traceback: {traceback.format_exc()}")
                payload_dict = {}
        else:
            # Parse as URL-encoded form data (application/x-www-form-urlencoded)
            try:
                from urllib.parse import parse_qs
                form_params = parse_qs(raw_body_str, keep_blank_values=True)
                payload_dict = {k: (v[0] if isinstance(v, list) and len(v) > 0 else v) for k, v in form_params.items()}
                logger.info(f"[Unitpay Payout Webhook] Parsed URL-encoded form data: {payload_dict}")
            except Exception as e:
                logger.error(f"[Unitpay Payout Webhook] Failed to parse URL-encoded: {e}", exc_info=True)
                payload_dict = {}
        
        # Extract fields with multiple fallback names
        amount = payload_dict.get("amount") or payload_dict.get("Amount") or ""
        apitxnid = payload_dict.get("apitxnid") or payload_dict.get("apitxnId") or payload_dict.get("apitxn_id") or ""
        refid = payload_dict.get("refid") or payload_dict.get("refId") or payload_dict.get("ref_id") or payload_dict.get("orderId") or payload_dict.get("order_id") or payload_dict.get("refno") or payload_dict.get("refNo") or ""
        status = payload_dict.get("status") or payload_dict.get("Status") or payload_dict.get("STATUS") or ""
        txnid = payload_dict.get("txnid") or payload_dict.get("txnId") or payload_dict.get("txn_id") or payload_dict.get("transactionId") or ""
        utr = payload_dict.get("utr") or payload_dict.get("Utr") or payload_dict.get("UTR") or payload_dict.get("bankutr") or payload_dict.get("bankUtr") or payload_dict.get("payid") or payload_dict.get("payId") or payload_dict.get("refno") or payload_dict.get("refNo") or ""
        
        # Log extracted fields
        logger.info(
            f"[Unitpay Payout Webhook] Extracted fields - "
            f"amount={amount}, refid={refid}, status={status}, "
            f"txnid={txnid}, apitxnid={apitxnid}, utr={utr}"
        )
        
        # Validate that we have at least apitxnid, refid, or txnid to find the transaction
        if not apitxnid and not refid and not txnid:
            logger.warning("[Unitpay Payout Webhook] Missing apitxnid, refid, and txnid - cannot find transaction")
            response = {
                "status": "error",
                "message": "Missing apitxnid, refid, and txnid - cannot find transaction",
                "orderId": ""
            }
            log_webhook_to_file("payout", payload_dict, response, error="Missing apitxnid/refid/txnid", provider="unitpay")
            return response
        
        # Initialize service
        service = PayoutService(db)
        
        # Convert amount to Decimal if provided
        amount_decimal = None
        if amount:
            try:
                amount_decimal = Decimal(str(amount))
            except Exception as e:
                logger.warning(f"[Unitpay Payout Webhook] Failed to parse amount '{amount}': {e}")
                amount_decimal = None
        
        # Use apitxnid as order_id (provider_reference_id) - this is what Unitpay uses to identify transactions
        # apitxnid from webhook = provider_reference_id in our database (stored when payout was created)
        # Fallback to txnid if apitxnid is not available, then refid
        order_id = apitxnid if apitxnid else (txnid if txnid else refid)
        
        if not order_id:
            error_msg = "Missing both txnid and refid - cannot find transaction"
            logger.error(f"[Unitpay Payout Webhook] {error_msg}")
            raise NotFoundError(error_msg)
        
        logger.info(f"[Unitpay Payout Webhook] Looking up transaction with order_id (provider_reference_id): {order_id}")
        
        # Handle webhook - this will update the payout transaction status
        updated_payout = service.handle_webhook(
            order_id=order_id,
            status=status.upper() if status else "PENDING",
            txn_ref_id=utr if utr else None,
            payout_id=apitxnid if apitxnid else None,
            amount=amount_decimal,
            db=db
        )
        
        logger.info(
            f"[Unitpay Payout Webhook] Updated payout txn: "
            f"txnid={updated_payout.txnid}, "
            f"status={updated_payout.status}, "
            f"rrn={updated_payout.rrn}, "
            f"provider_reference_id={updated_payout.provider_reference_id}"
        )
        
        # Prepare success response
        response = {
            "status": "success",
            "message": "Webhook processed successfully",
            "orderId": order_id
        }
        
        # Log webhook to file
        log_webhook_to_file("payout", payload_dict, response, provider="unitpay")
        
        # Send webhook to customer
        webhook_service = WebhookSenderService(db)
        webhook_result = webhook_service.send_webhook(
            service_code="payout",
            txnid=updated_payout.txnid,
            db=db
        )
        
        logger.info(f"[Unitpay Payout Webhook] Customer webhook sent: {webhook_result}")
        
        # Return 200 OK as required by Unitpay
        return response
        
    except NotFoundError as e:
        # Transaction not found - still return 200 to prevent retries
        error_msg = f"Transaction not found: {str(e)}"
        logger.error(f"[Unitpay Payout Webhook] {error_msg} - refid={refid}, txnid={txnid}")
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": refid if refid else txnid
        }
        log_webhook_to_file("payout", payload_dict, response, error=error_msg, provider="unitpay")
        return response
        
    except Exception as e:
        # Return 200 even on error to prevent webhook retries
        error_msg = f"Webhook processing failed: {str(e)}"
        logger.error(f"[Unitpay Payout Webhook] {error_msg} - refid={refid}, txnid={txnid}", exc_info=True)
        response = {
            "status": "error",
            "message": error_msg,
            "orderId": refid if refid else txnid
        }
        log_webhook_to_file("payout", payload_dict, response, error=error_msg, provider="unitpay")
        return response
