"""
Base repository with generic CRUD operations
"""

from typing import Generic, TypeVar, Type, Optional, List, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_
from datetime import datetime
from app.core.exceptions import NotFoundError
from app.common.mixins import SoftDeleteMixin

ModelType = TypeVar("ModelType")


class BaseRepository(Generic[ModelType]):
    """Base repository with generic CRUD operations"""

    def __init__(self, model: Type[ModelType], db: Session):
        """
        Initialize repository with model and database session
        
        Args:
            model: SQLAlchemy model class
            db: Database session
        """
        self.model = model
        self.db = db

    def get_by_id(self, id: str, include_deleted: bool = False) -> Optional[ModelType]:
        """
        Get a record by ID
        
        Args:
            id: Record ID
            include_deleted: Include soft-deleted records
            
        Returns:
            Model instance or None
        """
        query = self.db.query(self.model).filter(self.model.id == id)
        
        if hasattr(self.model, 'is_deleted') and not include_deleted:
            query = query.filter(self.model.is_deleted == False)
        
        return query.first()

    def get_all(
        self,
        skip: int = 0,
        limit: int = 100,
        include_deleted: bool = False,
        filters: Optional[Dict[str, Any]] = None
    ) -> List[ModelType]:
        """
        Get all records with pagination
        
        Args:
            skip: Number of records to skip
            limit: Maximum number of records to return
            include_deleted: Include soft-deleted records
            filters: Dictionary of filter conditions
            
        Returns:
            List of model instances
        """
        query = self.db.query(self.model)
        
        if hasattr(self.model, 'is_deleted') and not include_deleted:
            query = query.filter(self.model.is_deleted == False)
        
        if filters:
            for key, value in filters.items():
                if hasattr(self.model, key):
                    query = query.filter(getattr(self.model, key) == value)
        
        return query.offset(skip).limit(limit).all()

    def create(self, **kwargs) -> ModelType:
        """
        Create a new record
        
        Args:
            **kwargs: Model attributes
            
        Returns:
            Created model instance
        """
        db_obj = self.model(**kwargs)
        self.db.add(db_obj)
        self.db.commit()
        self.db.refresh(db_obj)
        return db_obj

    def update(self, id: str, **kwargs) -> ModelType:
        """
        Update a record by ID
        
        Args:
            id: Record ID
            **kwargs: Attributes to update
            
        Returns:
            Updated model instance
            
        Raises:
            NotFoundError: If record not found
        """
        db_obj = self.get_by_id(id)
        if not db_obj:
            raise NotFoundError(f"{self.model.__name__}")
        
        for key, value in kwargs.items():
            if hasattr(db_obj, key):
                setattr(db_obj, key, value)
        
        self.db.commit()
        self.db.refresh(db_obj)
        return db_obj

    def delete(self, id: str, hard: bool = False) -> bool:
        """
        Delete a record (soft delete if available, otherwise hard delete)
        
        Args:
            id: Record ID
            hard: Force hard delete even if soft delete is available
            
        Returns:
            True if deleted successfully
            
        Raises:
            NotFoundError: If record not found
        """
        db_obj = self.get_by_id(id, include_deleted=True)
        if not db_obj:
            raise NotFoundError(f"{self.model.__name__}")
        
        if isinstance(db_obj, SoftDeleteMixin) and not hard:
            db_obj.soft_delete()
            self.db.commit()
        else:
            self.db.delete(db_obj)
            self.db.commit()
        
        return True

    def update_status(self, id: str, status: str) -> ModelType:
        """
        Update record status
        
        Args:
            id: Record ID
            status: New status value
            
        Returns:
            Updated model instance
            
        Raises:
            NotFoundError: If record not found
        """
        return self.update(id, status=status)

    def count(self, include_deleted: bool = False, filters: Optional[Dict[str, Any]] = None) -> int:
        """
        Count records
        
        Args:
            include_deleted: Include soft-deleted records
            filters: Dictionary of filter conditions
            
        Returns:
            Count of records
        """
        query = self.db.query(self.model)
        
        if hasattr(self.model, 'is_deleted') and not include_deleted:
            query = query.filter(self.model.is_deleted == False)
        
        if filters:
            for key, value in filters.items():
                if hasattr(self.model, key):
                    query = query.filter(getattr(self.model, key) == value)
        
        return query.count()

    def exists(self, id: str) -> bool:
        """
        Check if record exists
        
        Args:
            id: Record ID
            
        Returns:
            True if exists, False otherwise
        """
        return self.get_by_id(id) is not None

