"""
Base service with common business logic
"""

from typing import Generic, TypeVar, Type, Optional, Dict, Any
from sqlalchemy.orm import Session
from app.common.base_repository import BaseRepository
from app.core.exceptions import ValidationError, NotFoundError, ConflictError

ModelType = TypeVar("ModelType")
CreateSchemaType = TypeVar("CreateSchemaType")
UpdateSchemaType = TypeVar("UpdateSchemaType")


class BaseService(Generic[ModelType, CreateSchemaType, UpdateSchemaType]):
    """Base service with common business logic"""

    def __init__(self, repository: BaseRepository[ModelType]):
        """
        Initialize service with repository
        
        Args:
            repository: Repository instance
        """
        self.repository = repository

    def validate_create(self, data: CreateSchemaType, db: Session) -> Dict[str, Any]:
        """
        Validate data before creation
        
        Args:
            data: Pydantic schema with creation data
            db: Database session
            
        Returns:
            Validated data as dictionary
            
        Raises:
            ValidationError: If validation fails
        """
        if isinstance(data, dict):
            return data
        return data.model_dump(exclude_unset=True)

    def validate_update(self, data: UpdateSchemaType, db: Session) -> Dict[str, Any]:
        """
        Validate data before update
        
        Args:
            data: Pydantic schema with update data
            db: Database session
            
        Returns:
            Validated data as dictionary
            
        Raises:
            ValidationError: If validation fails
        """
        if isinstance(data, dict):
            return data
        return data.model_dump(exclude_unset=True)

    def create(self, data: CreateSchemaType, db: Session) -> ModelType:
        """
        Create a new record
        
        Args:
            data: Pydantic schema with creation data
            db: Database session
            
        Returns:
            Created model instance
        """
        validated_data = self.validate_create(data, db)
        return self.repository.create(**validated_data)

    def update(self, id: str, data: UpdateSchemaType, db: Session) -> ModelType:
        """
        Update a record
        
        Args:
            id: Record ID
            data: Pydantic schema with update data
            db: Database session
            
        Returns:
            Updated model instance
        """
        validated_data = self.validate_update(data, db)
        return self.repository.update(id, **validated_data)

    def change_status(self, id: str, status: str, db: Session) -> ModelType:
        """
        Change record status
        
        Args:
            id: Record ID
            status: New status value
            db: Database session
            
        Returns:
            Updated model instance
        """
        return self.repository.update_status(id, status)

    def get_by_id(self, id: str, db: Session) -> Optional[ModelType]:
        """
        Get record by ID
        
        Args:
            id: Record ID
            db: Database session
            
        Returns:
            Model instance or None
        """
        return self.repository.get_by_id(id)

    def get_all(
        self,
        skip: int = 0,
        limit: int = 100,
        db: Session = None,
        filters: Optional[Dict[str, Any]] = None
    ) -> list[ModelType]:
        """
        Get all records with pagination
        
        Args:
            skip: Number of records to skip
            limit: Maximum number of records to return
            db: Database session (not used but kept for consistency)
            filters: Dictionary of filter conditions
            
        Returns:
            List of model instances
        """
        return self.repository.get_all(skip=skip, limit=limit, filters=filters)

