"""
KYC Configuration - Ownership types and document requirements
"""

import json
import os
from typing import Dict, List, Optional
from pathlib import Path


def load_kyc_config() -> Dict:
    """Load KYC configuration from JSON file"""
    config_path = Path(__file__).parent / "kyc_config.json"
    with open(config_path, 'r', encoding='utf-8') as f:
        return json.load(f)


def get_ownership_types() -> Dict[str, Dict]:
    """Get all ownership types"""
    config = load_kyc_config()
    return config.get("ownership_types", {})


def get_ownership_type(ownership_type: str) -> Optional[Dict]:
    """Get specific ownership type configuration"""
    ownership_types = get_ownership_types()
    return ownership_types.get(ownership_type)


def get_required_documents(ownership_type: str) -> List[str]:
    """Get list of all required documents for an ownership type"""
    config = get_ownership_type(ownership_type)
    if not config:
        return []
    
    common_required = config.get("common_documents", {}).get("required", [])
    specific_required = config.get("specific_documents", {}).get("required", [])
    
    return common_required + specific_required


def get_optional_documents(ownership_type: str) -> List[str]:
    """Get list of all optional documents for an ownership type"""
    config = get_ownership_type(ownership_type)
    if not config:
        return []
    
    common_optional = config.get("common_documents", {}).get("optional", [])
    specific_optional = config.get("specific_documents", {}).get("optional", [])
    
    return common_optional + specific_optional


def get_all_documents(ownership_type: str) -> Dict[str, List[str]]:
    """Get all documents (required and optional) for an ownership type"""
    return {
        "required": get_required_documents(ownership_type),
        "optional": get_optional_documents(ownership_type)
    }


def get_document_description(document_type: str) -> Optional[str]:
    """Get description for a document type"""
    config = load_kyc_config()
    descriptions = config.get("document_descriptions", {})
    return descriptions.get(document_type)


def is_valid_ownership_type(ownership_type: str) -> bool:
    """Check if ownership type is valid"""
    return ownership_type in get_ownership_types()

