"""
Application configuration management
"""

from typing import List
from pydantic_settings import BaseSettings
from pydantic import Field


class Settings(BaseSettings):
    """Application settings loaded from environment variables"""

    # Application
    APP_NAME: str = "Fintech Backend"
    APP_VERSION: str = "1.0.0"
    DEBUG: bool = False
    ENVIRONMENT: str = "production"

    # Server
    HOST: str = "0.0.0.0"
    PORT: int = 8000

    # Database
    DATABASE_URL: str = Field(
        ...,
        description="PostgreSQL database connection URL"
    )
    DATABASE_POOL_SIZE: int = 10
    DATABASE_MAX_OVERFLOW: int = 20

    # JWT
    SECRET_KEY: str = Field(
        ...,
        min_length=32,
        description="Secret key for JWT token signing"
    )
    ALGORITHM: str = "HS256"
    ACCESS_TOKEN_EXPIRE_MINUTES: int = 1440  # 24 hours

    # Security
    BCRYPT_ROUNDS: int = 12
    CORS_ORIGINS: str = "http://localhost:3000,http://localhost:8080,https://panel.gompay.in,https://api.gompay.in,http://ftech.kitpay.online"

    # File Upload
    MAX_UPLOAD_SIZE_MB: int = 10
    UPLOAD_DIR: str = "./uploads"

    # Rupeeflow API
    RUPEEFLOW_API_KEY: str = Field(
        default="",
        description="Rupeeflow API key"
    )
    RUPEEFLOW_SECRET_KEY: str = Field(
        default="",
        description="Rupeeflow secret key"
    )
    RUPEEFLOW_API_URL: str = Field(
        default="https://api.rupeeflow.co/api/v1/payments/v2/payout/create",
        description="Rupeeflow payout create endpoint URL"
    )
    RUPEEFLOW_STATUS_URL: str = Field(
        default="https://api.rupeeflow.co/api/v1/payments/payout/status",
        description="Rupeeflow payout status check endpoint URL"
    )
    RUPEEFLOW_PAYIN_URL: str = Field(
        default="https://api.rupeeflow.co/api/v1/payments/payin/create",
        # default="https://api.rupeeflow.co/api/v1/payments/v4/payin/create",
        description="Rupeeflow payin create endpoint URL"
    )

    # Unitpay API
    UNITPAY_TOKEN: str = Field(
        default="",
        description="Unitpay API token"
    )
    UNITPAY_API_URL: str = Field(
        default="https://unitpaygo.com/v/qr/collection",
        description="Unitpay payin create endpoint URL"
    )
    UNITPAY_WEBHOOK_BASE_URL: str = Field(
        default="https://api.gompay.in",
        description="Base URL for Unitpay webhook callbacks"
    )
    UNITPAY_PAYOUT_URL: str = Field(
        default="https://unitpaygo.com/v/payout/v2/transaction",
        description="Unitpay payout create endpoint URL"
    )

    # MytPiPay (myt) API – UPI collection
    MYT_API_TOKEN: str = Field(
        default="",
        description="MytPiPay API token for UPI collection"
    )
    MYT_PAYIN_URL: str = Field(
        default="https://banking.mytpipay.com/api/upi-collection/createorder",
        description="MytPiPay payin (create order) endpoint URL"
    )
    PAYIN_MIN_AMOUNT: float = Field(
        default=10.0,
        description="Minimum payin amount (configurable via env)"
    )

    # Kafka Configuration
    KAFKA_BOOTSTRAP_SERVERS: str = Field(
        default="localhost:9092",
        description="Kafka bootstrap servers (comma-separated)"
    )
    KAFKA_PAYIN_CREDIT_TOPIC: str = Field(
        default="payin-credit-delayed",
        description="Kafka topic for delayed payin wallet credits"
    )
    KAFKA_CONSUMER_GROUP_ID: str = Field(
        default="payin-credit-consumer-group",
        description="Kafka consumer group ID"
    )
    PAYIN_CREDIT_DELAY_SECONDS: int = Field(
        default=3600,
        description="Delay in seconds before crediting wallet after payin success (default: 1 hour)"
    )
    KAFKA_BATCH_SIZE: int = Field(
        default=1000,
        description="Kafka batch size for high throughput"
    )
    KAFKA_LINGER_MS: int = Field(
        default=10,
        description="Kafka producer linger time in milliseconds"
    )

    @property
    def cors_origins_list(self) -> List[str]:
        """Convert CORS_ORIGINS string to list"""
        return [origin.strip() for origin in self.CORS_ORIGINS.split(",")]

    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        case_sensitive = True


# Global settings instance
settings = Settings()

