"""
Dependency injection functions
"""

from typing import Optional, Literal
from fastapi import Depends, HTTPException, status, Header
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from sqlalchemy.orm import Session
from app.core.database import get_db
from app.core.security import decode_token
from app.core.exceptions import UnauthorizedError, ForbiddenError
from app.models.user import User
from app.models.admin import Admin
from app.services.api_key_service import APIKeyService

# Security schemes for Swagger UI
security_scheme_user = HTTPBearer(
    scheme_name="BearerAuthUser",
    bearerFormat="JWT",
    description="User JWT Token - Get token from /users/login"
)
security_scheme_admin = HTTPBearer(
    scheme_name="BearerAuthAdmin",
    bearerFormat="JWT",
    description="Admin JWT Token - Get token from /admin/login"
)


async def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security_scheme_user),
    db: Session = Depends(get_db)
) -> User:
    """
    Dependency to get current authenticated user from JWT token
    """
    token = credentials.credentials
    
    payload = decode_token(token)
    user_id: Optional[str] = payload.get("sub")
    role: Optional[str] = payload.get("role")
    
    if not user_id or role != "user":
        raise UnauthorizedError("Invalid token payload")
    
    user = db.query(User).filter(User.id == user_id, User.deleted_at.is_(None)).first()
    if not user:
        raise UnauthorizedError("User not found")
    
    return user


async def get_current_admin(
    credentials: HTTPAuthorizationCredentials = Depends(security_scheme_admin),
    db: Session = Depends(get_db)
) -> Admin:
    """
    Dependency to get current authenticated admin from JWT token
    """
    token = credentials.credentials
    
    payload = decode_token(token)
    admin_id: Optional[str] = payload.get("sub")
    role: Optional[str] = payload.get("role")
    
    if not admin_id or role != "admin":
        raise UnauthorizedError("Invalid token payload")
    
    admin = db.query(Admin).filter(Admin.id == admin_id).first()
    if not admin:
        raise UnauthorizedError("Admin not found")
    
    return admin


def require_role(required_role: Literal["admin", "user"]):
    """
    Factory function to create role-based dependency
    """
    if required_role == "admin":
        return get_current_admin
    else:
        return get_current_user


async def get_current_user_by_api_key(
    x_api_key: str = Header(..., alias="X-API-Key", description="API Key"),
    db: Session = Depends(get_db)
) -> User:
    """
    Dependency to get current authenticated user from API key
    """
    # Verify API key
    api_key_service = APIKeyService(db)
    api_key_obj = api_key_service.verify_api_key(x_api_key)
    
    if not api_key_obj:
        raise UnauthorizedError("Invalid API key")
    
    # Get user
    user = db.query(User).filter(User.id == api_key_obj.user_id, User.deleted_at.is_(None)).first()
    if not user:
        raise UnauthorizedError("User not found")
    
    if not user.is_active:
        raise ForbiddenError("User account is inactive")
    
    return user

