"""
Security utilities for JWT and password hashing
"""

import bcrypt
import hmac
import hashlib
import json
import secrets
import string
from datetime import datetime, timedelta
from typing import Optional, Dict, Any
from jose import JWTError, jwt
from cryptography.fernet import Fernet
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
from cryptography.hazmat.backends import default_backend
import base64
from app.core.config import settings
from app.core.exceptions import UnauthorizedError


def verify_password(plain_password: str, hashed_password: str) -> bool:
    """Verify a password against its hash"""
    try:
        return bcrypt.checkpw(plain_password.encode('utf-8'), hashed_password.encode('utf-8'))
    except Exception:
        return False


def get_password_hash(password: str) -> str:
    """Hash a password using bcrypt"""
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt(rounds=12)).decode('utf-8')


def create_access_token(data: Dict[str, Any], expires_delta: Optional[timedelta] = None) -> str:
    """Create a JWT access token"""
    to_encode = data.copy()
    
    if expires_delta:
        expire = datetime.utcnow() + expires_delta
    else:
        expire = datetime.utcnow() + timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    
    to_encode.update({"exp": expire, "iat": datetime.utcnow()})
    encoded_jwt = jwt.encode(to_encode, settings.SECRET_KEY, algorithm=settings.ALGORITHM)
    return encoded_jwt


def decode_token(token: str) -> Dict[str, Any]:
    """Decode and verify a JWT token"""
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=[settings.ALGORITHM])
        return payload
    except JWTError:
        raise UnauthorizedError("Invalid authentication credentials")


def generate_webhook_secret() -> str:
    """Generate a random webhook secret for signing"""
    alphabet = string.ascii_letters + string.digits
    return ''.join(secrets.choice(alphabet) for _ in range(64))


def sign_webhook_payload(payload: Dict[str, Any], secret: str) -> str:
    """
    Sign webhook payload using HMAC-SHA256
    
    Signature Formula:
    1. Extract user_reference_id, txnid, and status from payload
    2. Create signature string: user_reference_id|txnid|status
    3. Compute HMAC-SHA256 hash using the secret
    4. Return hex-encoded signature
    
    Args:
        payload: Webhook payload dictionary
        secret: Webhook secret key
        
    Returns:
        Hex-encoded signature string
    """
    # Extract values from payload
    user_reference_id = payload.get('user_reference_id', '')
    data = payload.get('data', {})
    txnid = data.get('txnid', '') if isinstance(data, dict) else ''
    status = data.get('status', '') if isinstance(data, dict) else ''
    
    # Create signature string: user_reference_id|txnid|status
    signature_string = f"{user_reference_id}|{txnid}|{status}"
    
    # Compute HMAC-SHA256
    signature = hmac.new(
        secret.encode('utf-8'),
        signature_string.encode('utf-8'),
        hashlib.sha256
    ).hexdigest()
    
    return signature


def verify_webhook_signature(payload: Dict[str, Any], signature: str, secret: str) -> bool:
    """
    Verify webhook signature
    
    Args:
        payload: Webhook payload dictionary
        signature: Signature to verify (hex-encoded)
        secret: Webhook secret key
        
    Returns:
        True if signature is valid, False otherwise
    """
    expected_signature = sign_webhook_payload(payload, secret)
    return hmac.compare_digest(expected_signature, signature)


def _get_encryption_key() -> bytes:
    """Get encryption key derived from SECRET_KEY"""
    # Use PBKDF2 to derive a 32-byte key from SECRET_KEY
    kdf = PBKDF2HMAC(
        algorithm=hashes.SHA256(),
        length=32,
        salt=b'fintech_webhook_salt',  # Fixed salt for consistency
        iterations=100000,
        backend=default_backend()
    )
    key = base64.urlsafe_b64encode(kdf.derive(settings.SECRET_KEY.encode()))
    return key


def encrypt_api_secret(plain_secret: str) -> str:
    """
    Encrypt API secret for storage (can be decrypted for webhook signing)
    
    Args:
        plain_secret: Plain API secret
        
    Returns:
        Encrypted secret string
    """
    key = _get_encryption_key()
    f = Fernet(key)
    encrypted = f.encrypt(plain_secret.encode('utf-8'))
    return encrypted.decode('utf-8')


def decrypt_api_secret(encrypted_secret: str) -> str:
    """
    Decrypt API secret for webhook signing
    
    Args:
        encrypted_secret: Encrypted API secret
        
    Returns:
        Plain API secret string
    """
    key = _get_encryption_key()
    f = Fernet(key)
    decrypted = f.decrypt(encrypted_secret.encode('utf-8'))
    return decrypted.decode('utf-8')

