"""
FastAPI application entry point
"""

from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse, FileResponse, StreamingResponse
from fastapi.openapi.utils import get_openapi
from fastapi.exceptions import RequestValidationError
from pydantic import ValidationError
import os
import logging
import mimetypes

from app.core.config import settings

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
from app.core.database import engine, Base
from app.core.exceptions import BaseAPIException
from app.api.user import routes as user_routes
from app.api.admin import routes as admin_routes
from app.api.service import routes as service_routes
from app.api.webhook import routes as webhook_routes

# Create database tables (in production, use Alembic migrations)
Base.metadata.create_all(bind=engine)

# Create FastAPI app
app = FastAPI(
    title=settings.APP_NAME,
    version=settings.APP_VERSION,
    debug=settings.DEBUG
)


def custom_openapi():
    """Custom OpenAPI schema with security schemes"""
    if app.openapi_schema:
        return app.openapi_schema
    
    openapi_schema = get_openapi(
        title=settings.APP_NAME,
        version=settings.APP_VERSION,
        description="Fintech Backend API with JWT Authentication",
        routes=app.routes,
    )
    
    # Ensure components exist
    if "components" not in openapi_schema:
        openapi_schema["components"] = {}
    
    # Security schemes will be automatically added by HTTPBearer
    # But we ensure components exist
    if "securitySchemes" not in openapi_schema["components"]:
        openapi_schema["components"]["securitySchemes"] = {}
    
    app.openapi_schema = openapi_schema
    return app.openapi_schema


app.openapi = custom_openapi

# CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.cors_origins_list,
    allow_origin_regex=r"http://localhost:\d+|http://127\.0\.0\.1:\d+|https?://([a-zA-Z0-9-]+\.)*gompay\.in$|https?://([a-zA-Z0-9-]+\.)*kitpay\.online$",
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)


# Global exception handler
@app.exception_handler(BaseAPIException)
async def api_exception_handler(request, exc: BaseAPIException):
    """Handle custom API exceptions"""
    return JSONResponse(
        status_code=exc.status_code,
        content={"detail": exc.detail}
    )

# Exception handler for validation errors - log everything for webhook endpoints
@app.exception_handler(RequestValidationError)
async def validation_exception_handler(request: Request, exc: RequestValidationError):
    """Handle validation errors - for webhooks, just log and return 200"""
    import logging
    logger = logging.getLogger(__name__)
    
    # If this is a webhook endpoint, log everything and return 200
    if "/webhooks/" in str(request.url.path):
        try:
            # Try to read body if available
            body = await request.body()
            body_str = body.decode('utf-8', errors='ignore') if body else ""
        except:
            body_str = "<body already consumed>"
        
        logger.info("=" * 80)
        logger.info(f"[Webhook Validation Error] Path: {request.url.path}")
        logger.info(f"[Webhook Validation Error] Validation Errors: {exc.errors()}")
        logger.info(f"[Webhook Validation Error] Raw Body: {body_str}")
        logger.info(f"[Webhook Validation Error] Headers: {dict(request.headers)}")
        logger.info("=" * 80)
        
        # Return 200 so webhook provider doesn't retry
        return JSONResponse(
            status_code=200,
            content={
                "status": "success",
                "message": "Webhook received (validation bypassed - check logs)"
            }
        )
    
    # For other endpoints, return normal 422
    return JSONResponse(
        status_code=422,
        content={"detail": exc.errors()}
    )



# Register WebP MIME type if not already registered
if not mimetypes.guess_type("test.webp")[0]:
    mimetypes.add_type("image/webp", ".webp")

# Setup uploads directory
uploads_dir = "uploads"
os.makedirs(uploads_dir, exist_ok=True)

# Add explicit route for static files to ensure correct Content-Type headers
# Using StreamingResponse instead of FileResponse to fix WebP corruption issues
@app.get("/static/{file_path:path}")
async def serve_static_file(file_path: str):
    """Serve static files with correct Content-Type headers"""
    file_full_path = os.path.join(uploads_dir, file_path)
    
    if not os.path.isfile(file_full_path):
        return JSONResponse(status_code=404, content={"detail": "File not found"})
    
    # Force correct mime type - explicitly set WebP
    if file_path.endswith(".webp"):
        media_type = "image/webp"
    else:
        media_type, _ = mimetypes.guess_type(file_full_path)
        media_type = media_type or "application/octet-stream"
    
    # 🔥 STREAM instead of FileResponse (fixes WebP corruption issues)
    def iterfile():
        with open(file_full_path, "rb") as f:
            yield from f
    
    return StreamingResponse(
        iterfile(),
        media_type=media_type,
        headers={
            "Content-Disposition": f'inline; filename="{os.path.basename(file_full_path)}"',
            "Cache-Control": "no-store"
        }
    )

# Note: We use a route handler instead of mount to ensure correct Content-Type headers
# The route handler above will serve all static files with proper MIME types

# Include routers
app.include_router(user_routes.router)
app.include_router(admin_routes.router)
app.include_router(service_routes.router)
app.include_router(webhook_routes.router)



@app.get("/")
def root():
    """Root endpoint"""
    return {
        "message": f"Welcome to {settings.APP_NAME}",
        "version": settings.APP_VERSION
    }


@app.get("/health")
def health_check():
    """Health check endpoint"""
    return {"status": "healthy"}

