"""
Charge repository
"""

from typing import Optional, List
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_
from decimal import Decimal
from app.common.base_repository import BaseRepository
from app.models.charge import Charge


class ChargeRepository(BaseRepository[Charge]):
    """Charge repository"""

    def __init__(self, db: Session):
        super().__init__(Charge, db)

    def get_by_service_id(self, service_id: str, user_id: Optional[str] = None) -> List[Charge]:
        """Get all charges for a service, optionally filtered by user_id"""
        query = self.db.query(self.model).filter(
            self.model.service_id == service_id
        )
        
        if user_id:
            # Get user-specific charges or global charges (user_id is NULL)
            query = query.filter(
                or_(
                    self.model.user_id == user_id,
                    self.model.user_id.is_(None)
                )
            )
        else:
            # Get all charges for the service
            pass
        
        return query.order_by(self.model.from_amount.asc()).all()
    
    def get_by_user_and_service(self, user_id: str, service_id: str) -> List[Charge]:
        """Get charges for a specific user and service"""
        return self.db.query(self.model).filter(
            and_(
                self.model.service_id == service_id,
                or_(
                    self.model.user_id == user_id,
                    self.model.user_id.is_(None)  # Include global charges
                )
            )
        ).order_by(self.model.from_amount.asc()).all()
    
    def get_overlapping_charges(
        self,
        service_id: str,
        from_amount: Decimal,
        to_amount: Decimal,
        user_id: Optional[str] = None,
        exclude_id: Optional[int] = None
    ) -> List[Charge]:
        """
        Get charges that overlap with the given amount range
        
        Args:
            service_id: Service ID
            from_amount: Start of amount range
            to_amount: End of amount range (0 means unlimited)
            user_id: User ID (optional, for user-specific charges)
            exclude_id: Charge ID to exclude (for updates)
            
        Returns:
            List of overlapping charges
        """
        query = self.db.query(self.model).filter(
            self.model.service_id == service_id
        )
        
        # Filter by user_id if provided
        if user_id:
            query = query.filter(
                or_(
                    self.model.user_id == user_id,
                    self.model.user_id.is_(None)
                )
            )
        
        # Exclude current charge if updating
        if exclude_id:
            query = query.filter(self.model.id != exclude_id)
        
        # Check for overlaps
        # Case 1: Universal charge (from=0, to=0) overlaps with everything
        # Case 2: New range overlaps with existing range
        overlapping = []
        charges = query.all()
        
        for charge in charges:
            # Universal charge (applies to all amounts)
            if charge.from_amount == Decimal('0.00') and charge.to_amount == Decimal('0.00'):
                overlapping.append(charge)
                continue
            
            # New charge is universal
            if from_amount == Decimal('0.00') and to_amount == Decimal('0.00'):
                overlapping.append(charge)
                continue
            
            # Check range overlap
            # Two ranges overlap if: new_from < existing_to AND new_to > existing_from
            # Handle unlimited upper bound (to_amount = 0 means unlimited)
            existing_to = charge.to_amount if charge.to_amount > Decimal('0.00') else Decimal('999999999.99')
            new_to = to_amount if to_amount > Decimal('0.00') else Decimal('999999999.99')
            
            if from_amount < existing_to and new_to > charge.from_amount:
                overlapping.append(charge)
        
        return overlapping

