"""
KYC Document repository
"""

from typing import Optional, List
from sqlalchemy.orm import Session
from sqlalchemy.dialects.postgresql import UUID
from app.common.base_repository import BaseRepository
from app.models.kyc_doc import KYCDoc


class KYCDocRepository(BaseRepository[KYCDoc]):
    """KYC Document repository"""

    def __init__(self, db: Session):
        super().__init__(KYCDoc, db)

    def get_by_user_and_document_type(self, user_id: UUID, document_type: str) -> Optional[KYCDoc]:
        """Get document by user_id and document_type"""
        return self.db.query(self.model).filter(
            self.model.user_id == user_id,
            self.model.document_type == document_type
        ).order_by(self.model.created_at.desc()).first()

    def get_by_kyc_id(self, kyc_id: UUID) -> List[KYCDoc]:
        """Get all documents for a KYC"""
        return self.db.query(self.model).filter(
            self.model.kyc_id == kyc_id
        ).all()

    def get_by_user_id(self, user_id: UUID) -> List[KYCDoc]:
        """Get all documents for a user"""
        return self.db.query(self.model).filter(
            self.model.user_id == user_id
        ).all()

