"""
Payout Transaction repository
"""

from typing import Optional, List
from sqlalchemy.orm import Session
from app.common.base_repository import BaseRepository
from app.models.payout_transaction import PayoutTransaction


class PayoutTransactionRepository(BaseRepository[PayoutTransaction]):
    """Payout Transaction repository"""

    def __init__(self, db: Session):
        super().__init__(PayoutTransaction, db)

    def get_by_user_id(self, user_id: str, skip: int = 0, limit: int = 100) -> List[PayoutTransaction]:
        """Get all payout transactions for a user"""
        return self.db.query(self.model).filter(
            self.model.user_id == user_id
        ).order_by(self.model.created_at.desc()).offset(skip).limit(limit).all()

    def get_by_txnid(self, txnid: str) -> Optional[PayoutTransaction]:
        """Get payout transaction by transaction ID"""
        return self.db.query(self.model).filter(
            self.model.txnid == txnid
        ).first()
    
    def get_by_provider_reference_id(self, provider_reference_id: str) -> Optional[PayoutTransaction]:
        """Get payout transaction by provider reference ID"""
        return self.db.query(self.model).filter(
            self.model.provider_reference_id == provider_reference_id
        ).first()
    
    def get_by_user_reference_id(self, user_reference_id: str) -> Optional[PayoutTransaction]:
        """Get payout transaction by user reference ID"""
        return self.db.query(self.model).filter(
            self.model.user_reference_id == user_reference_id
        ).first()
    
    def get_by_apitxnid_in_response(self, apitxnid: str) -> Optional[PayoutTransaction]:
        """Get payout transaction by apitxnid stored in api_response JSON field"""
        # Search for apitxnid in the api_response JSON field
        # This is useful when webhook arrives before provider_reference_id is set
        from sqlalchemy import text
        return self.db.query(self.model).filter(
            text("api_response->>'apitxnid' = :apitxnid").bindparam(apitxnid=apitxnid)
        ).first()

