"""
Wallet Transaction repository
"""

from typing import Optional, List
from sqlalchemy.orm import Session
from app.common.base_repository import BaseRepository
from app.models.wallet_transaction import WalletTransaction


class WalletTransactionRepository(BaseRepository[WalletTransaction]):
    """Wallet Transaction repository"""

    def __init__(self, db: Session):
        super().__init__(WalletTransaction, db)

    def get_by_user_id(self, user_id: str, skip: int = 0, limit: int = 100) -> List[WalletTransaction]:
        """Get all transactions for a user"""
        return self.db.query(self.model).filter(
            self.model.user_id == user_id
        ).order_by(self.model.created_at.desc()).offset(skip).limit(limit).all()

    def get_all(self, skip: int = 0, limit: int = 100, username: Optional[str] = None) -> List[WalletTransaction]:
        """Get all transactions, optionally filtered by username (email)"""
        from app.models.user import User
        
        query = self.db.query(self.model)
        
        # Join with users table if filtering by username
        if username:
            query = query.join(User, self.model.user_id == User.id).filter(
                User.email.ilike(f"%{username}%")
            )
        
        return query.order_by(
            self.model.created_at.desc()
        ).offset(skip).limit(limit).all()

    def get_by_wallet_id(self, wallet_id: str, skip: int = 0, limit: int = 100) -> List[WalletTransaction]:
        """Get all transactions for a wallet"""
        return self.db.query(self.model).filter(
            self.model.wallet_id == wallet_id
        ).order_by(self.model.created_at.desc()).offset(skip).limit(limit).all()

    def get_by_txnid(self, txnid: str) -> Optional[WalletTransaction]:
        """Get transaction by transaction ID"""
        return self.db.query(self.model).filter(
            self.model.txnid == txnid
        ).first()
