"""
Webhook Log repository
"""

from typing import Optional, List, Dict, Any
from sqlalchemy.orm import Session
from app.common.base_repository import BaseRepository
from app.models.webhook_log import WebhookLog


class WebhookLogRepository(BaseRepository[WebhookLog]):
    """Webhook Log repository"""

    def __init__(self, db: Session):
        super().__init__(WebhookLog, db)

    def get_by_user_id(
        self,
        user_id: str,
        skip: int = 0,
        limit: int = 100
    ) -> List[WebhookLog]:
        """Get webhook logs by user ID"""
        return self.db.query(self.model).filter(
            self.model.user_id == user_id
        ).order_by(
            self.model.created_at.desc()
        ).offset(skip).limit(limit).all()

    def get_by_service_id(
        self,
        service_id: str,
        skip: int = 0,
        limit: int = 100
    ) -> List[WebhookLog]:
        """Get webhook logs by service ID"""
        return self.db.query(self.model).filter(
            self.model.service_id == service_id
        ).order_by(
            self.model.created_at.desc()
        ).offset(skip).limit(limit).all()

    def get_by_txnid(self, txnid: str) -> List[WebhookLog]:
        """Get webhook logs by transaction ID"""
        return self.db.query(self.model).filter(
            self.model.txnid == txnid
        ).order_by(
            self.model.created_at.desc()
        ).all()

    def get_all_with_filters(
        self,
        skip: int = 0,
        limit: int = 100,
        user_id: Optional[str] = None,
        service_id: Optional[str] = None,
        txnid: Optional[str] = None
    ) -> List[WebhookLog]:
        """Get webhook logs with optional filters"""
        query = self.db.query(self.model)
        
        if user_id:
            query = query.filter(self.model.user_id == user_id)
        
        if service_id:
            query = query.filter(self.model.service_id == service_id)
        
        if txnid:
            query = query.filter(self.model.txnid == txnid)
        
        return query.order_by(
            self.model.created_at.desc()
        ).offset(skip).limit(limit).all()

    def count_with_filters(
        self,
        user_id: Optional[str] = None,
        service_id: Optional[str] = None,
        txnid: Optional[str] = None
    ) -> int:
        """Count webhook logs with optional filters"""
        query = self.db.query(self.model)
        
        if user_id:
            query = query.filter(self.model.user_id == user_id)
        
        if service_id:
            query = query.filter(self.model.service_id == service_id)
        
        if txnid:
            query = query.filter(self.model.txnid == txnid)
        
        return query.count()

