"""
Charge schemas
"""

from typing import Optional
from pydantic import BaseModel, Field, ConfigDict
from uuid import UUID
from datetime import datetime
from decimal import Decimal


class ChargeCreate(BaseModel):
    """Charge creation schema"""
    service_id: UUID = Field(..., description="Service ID")
    user_id: Optional[UUID] = Field(None, description="User ID (None for global charges)")
    from_amount: Decimal = Field(..., ge=0, description="Minimum transaction amount (0 for no limit)")
    to_amount: Decimal = Field(..., ge=0, description="Maximum transaction amount (0 for no limit/unlimited)")
    type: str = Field(..., description="Charge type: surcharge or commission")
    is_percent: bool = Field(..., description="True for percentage, False for fixed amount")
    amount: Decimal = Field(..., gt=0, description="Charge amount or percentage value")
    gst: Optional[Decimal] = Field(None, ge=0, le=100, description="GST percentage (null = default 18%, 0 = no GST)")


class ChargeUpdate(BaseModel):
    """Charge update schema"""
    service_id: Optional[UUID] = None
    user_id: Optional[UUID] = None
    from_amount: Optional[Decimal] = Field(None, ge=0)
    to_amount: Optional[Decimal] = Field(None, ge=0)
    type: Optional[str] = None
    is_percent: Optional[bool] = None
    amount: Optional[Decimal] = Field(None, gt=0)
    gst: Optional[Decimal] = Field(None, ge=0, le=100, description="GST percentage (null = default 18%, 0 = no GST)")


class ChargeResponse(BaseModel):
    """Charge response schema"""
    id: int
    user_id: Optional[UUID] = None
    service_id: UUID
    from_amount: Decimal
    to_amount: Decimal
    type: str
    is_percent: bool
    amount: Decimal
    gst: Optional[Decimal] = None
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class ChargeWithServiceResponse(BaseModel):
    """Charge response schema with service information"""
    id: int
    user_id: Optional[UUID] = None
    service_id: UUID
    service_name: Optional[str] = None
    service_code: Optional[str] = None
    from_amount: Decimal
    to_amount: Decimal
    type: str
    is_percent: bool
    amount: Decimal
    gst: Optional[Decimal] = None
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)

