"""
KYC schemas
"""

from typing import Optional, List, Dict, Any
from pydantic import BaseModel, Field, ConfigDict
from uuid import UUID
from datetime import datetime


class KYCCreate(BaseModel):
    """KYC creation schema - basic details only"""
    ownership_type: str = Field(..., description="Ownership type: individual, private_limited, public_limited, llp, partnership, sole_proprietorship, huf, trust, society")
    company_name: str = Field(..., min_length=1, description="Company name or individual name")
    date_of_birth: Optional[str] = Field(None, description="Date of birth (for individuals)")
    
    # Common identification numbers
    pan_number: Optional[str] = Field(None, min_length=10, max_length=10, description="PAN Number (10 characters)")
    aadhaar_number: Optional[str] = Field(None, min_length=12, max_length=12, description="Aadhaar Number (12 digits)")
    
    address: Optional[str] = None
    city: Optional[str] = None
    state: Optional[str] = None
    country: Optional[str] = None
    postal_code: Optional[str] = None
    notes: Optional[str] = None


class KYCUpdate(BaseModel):
    """KYC update schema"""
    ownership_type: Optional[str] = None
    document_type: Optional[str] = None
    document_number: Optional[str] = None
    company_name: Optional[str] = None
    date_of_birth: Optional[str] = None
    pan_number: Optional[str] = Field(None, min_length=10, max_length=10)
    aadhaar_number: Optional[str] = Field(None, min_length=12, max_length=12)
    address: Optional[str] = None
    city: Optional[str] = None
    state: Optional[str] = None
    country: Optional[str] = None
    postal_code: Optional[str] = None
    notes: Optional[str] = None
    status: Optional[str] = None


class KYCResponse(BaseModel):
    """KYC response schema"""
    id: UUID
    user_id: UUID
    ownership_type: str
    document_type: Optional[str]
    document_number: Optional[str]
    company_name: str
    date_of_birth: Optional[str]
    pan_number: Optional[str]
    aadhaar_number: Optional[str]
    address: Optional[str]
    city: Optional[str]
    state: Optional[str]
    country: Optional[str]
    postal_code: Optional[str]
    status: str
    notes: Optional[str]
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class KYCStatusResponse(BaseModel):
    """KYC status response schema"""
    status: str  # not_submitted, pending, under_review, completed, rejected
    kyc: Optional[KYCResponse] = None
    
    model_config = ConfigDict(from_attributes=True)


class OwnershipTypeInfo(BaseModel):
    """Ownership type information schema"""
    name: str
    description: str
    common_documents: Dict[str, List[str]]
    specific_documents: Dict[str, List[str]]
    
    model_config = ConfigDict(from_attributes=True)


class KYCConfigResponse(BaseModel):
    """KYC configuration response schema"""
    ownership_types: Dict[str, OwnershipTypeInfo]
    document_descriptions: Dict[str, str]
    
    model_config = ConfigDict(from_attributes=True)


class DocumentStatusInfo(BaseModel):
    """Document status information schema"""
    document_type: str
    document_name: str
    description: Optional[str] = None
    is_required: bool
    status: str  # not_submitted, submitted, pending, approved, rejected
    document_number: Optional[str] = None  # Document number if submitted
    doc_id: Optional[UUID] = None
    submitted_at: Optional[datetime] = None
    
    model_config = ConfigDict(from_attributes=True)


class RequiredDocumentsResponse(BaseModel):
    """Required documents response schema"""
    ownership_type: str
    ownership_type_name: str
    required_documents: List[DocumentStatusInfo]
    optional_documents: List[DocumentStatusInfo]
    
    model_config = ConfigDict(from_attributes=True)
