"""
Payin Transaction schemas
"""

from typing import Optional, Dict, Any
from pydantic import BaseModel, Field, ConfigDict, model_validator
from uuid import UUID
from datetime import datetime
from decimal import Decimal


class PayinTransactionCreate(BaseModel):
    """Payin Transaction creation schema"""
    amount: Decimal = Field(..., gt=0, description="Payin amount")
    name: str = Field(..., min_length=1, max_length=255, description="Payer name")

    @model_validator(mode="after")
    def check_min_amount(self):
        from app.core.config import settings
        min_amount = Decimal(str(settings.PAYIN_MIN_AMOUNT))
        if self.amount < min_amount:
            raise ValueError(f"Amount must be at least {min_amount}")
        return self
    email: str = Field(..., description="Payer email")
    mobile: str = Field(..., min_length=10, max_length=15, description="Payer mobile number")
    vpa: Optional[str] = Field(None, max_length=255, description="Payer VPA (Virtual Payment Address)")
    reference_id: str = Field(..., min_length=1, max_length=255, description="User-provided reference ID")


class PayinTransactionResponse(BaseModel):
    """Payin Transaction response schema (for users - excludes api_response)"""
    id: int
    user_id: UUID
    txnid: str
    user_reference_id: Optional[str] = None
    amount: Decimal
    charge: Decimal
    payee_vpa: Optional[str] = None
    payee_name: Optional[str] = None
    status: str
    qr_text: Optional[str] = None
    payment_url: Optional[str] = None
    api_provider: Optional[str] = None
    provider_reference_id: Optional[str] = None
    refunded: bool
    rrn: Optional[str] = None
    settled: bool = Field(default=False, description="Whether the wallet has been credited (settled)")
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class PayinInitiateResponse(BaseModel):
    """Payin initiate API response schema for third-party users"""
    status: str = Field(..., description="Status: success, error, failed, pending")
    msg: str = Field(..., description="Response message")
    data: Dict[str, Any] = Field(default_factory=dict, description="Response data")


class PayinStatusResponse(BaseModel):
    """Payin status API response schema for third-party users"""
    status: str = Field(..., description="Status: success, error, failed, pending")
    msg: str = Field(..., description="Response message")
    data: Dict[str, Any] = Field(default_factory=dict, description="Response data")


class PayinTransactionWithUserResponse(BaseModel):
    """Payin Transaction response schema with user information (for admin)"""
    id: int
    user_id: UUID
    txnid: str
    user_reference_id: Optional[str] = None
    amount: Decimal
    charge: Decimal
    payee_vpa: Optional[str] = None
    payee_name: Optional[str] = None
    status: str
    api_response: Optional[Dict[str, Any]] = None
    qr_text: Optional[str] = None
    payment_url: Optional[str] = None
    api_provider: Optional[str] = None
    provider_reference_id: Optional[str] = None
    refunded: bool
    user_name: Optional[str] = None  # User's full name
    username: Optional[str] = None  # User's email (used as username)
    rrn: Optional[str] = None
    settled: bool = Field(default=False, description="Whether the wallet has been credited (settled)")
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class PayinStatusUpdateRequest(BaseModel):
    """Admin request schema for updating payin transaction status"""
    status: str = Field(..., description="New status: success, failed, pending, error")
    credit_amount: bool = Field(default=False, description="Whether to credit the wallet (only if status is success)")
    send_webhook: bool = Field(default=False, description="Whether to send webhook notification")


class UnsettledBalanceResponse(BaseModel):
    """Unsettled balance response schema"""
    total_unsettled_amount: Decimal = Field(..., description="Total amount of successful payin transactions that are not settled")
    total_unsettled_count: int = Field(..., description="Total count of successful payin transactions that are not settled")
    user_id: Optional[UUID] = Field(None, description="User ID if filtered by user")


class PayinStatsResponse(BaseModel):
    """Payin statistics response schema"""
    today: dict = Field(..., description="Today's statistics")
    current_month: dict = Field(..., description="Current month statistics")
    total_till_today: dict = Field(..., description="Total statistics till today")

