"""
Payout Transaction schemas
"""

from typing import Optional, Dict, Any
from pydantic import BaseModel, Field, ConfigDict
from uuid import UUID
from datetime import datetime
from decimal import Decimal


class PayoutTransactionCreate(BaseModel):
    """Payout Transaction creation schema"""
    amount: Decimal = Field(..., gt=0, description="Payout amount")
    bene_name: str = Field(..., min_length=1, max_length=255, description="Beneficiary name")
    bene_ifsc: str = Field(..., min_length=1, max_length=20, description="Beneficiary IFSC code")
    bene_acc_no: str = Field(..., min_length=1, max_length=50, description="Beneficiary account number")
    reference_id: str = Field(..., min_length=1, max_length=255, description="User-provided reference ID")


class PayoutTransactionResponse(BaseModel):
    """Payout Transaction response schema (for users - excludes api_response)"""
    id: UUID
    user_id: UUID
    txnid: str
    user_reference_id: Optional[str] = None
    amount: Decimal
    charge: Decimal
    bene_name: str
    bene_ifsc: str
    bene_acc_no: str
    status: str
    api_provider: Optional[str] = None
    provider_reference_id: Optional[str] = None
    refunded: bool
    rrn: Optional[str] = None
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class PayoutInitiateResponse(BaseModel):
    """Payout initiate API response schema for third-party users"""
    status: str = Field(..., description="Status: success, error, failed, pending")
    msg: str = Field(..., description="Response message")
    data: Dict[str, Any] = Field(default_factory=dict, description="Response data")


class PayoutStatusResponse(BaseModel):
    """Payout status API response schema for third-party users"""
    status: str = Field(..., description="Status: success, error, failed, pending")
    msg: str = Field(..., description="Response message")
    data: Dict[str, Any] = Field(default_factory=dict, description="Response data")


class PayoutTransactionWithUserResponse(BaseModel):
    """Payout Transaction response schema with user information (for admin)"""
    id: UUID
    user_id: UUID
    txnid: str
    user_reference_id: Optional[str] = None
    amount: Decimal
    charge: Decimal
    bene_name: str
    bene_ifsc: str
    bene_acc_no: str
    status: str
    api_response: Optional[Dict[str, Any]] = None
    api_provider: Optional[str] = None
    provider_reference_id: Optional[str] = None
    refunded: bool
    user_name: Optional[str] = None  # User's full name
    username: Optional[str] = None  # User's email (used as username)
    rrn: Optional[str] = None
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class PayoutStatusUpdateRequest(BaseModel):
    """Admin request schema for updating payout transaction status"""
    status: str = Field(..., description="New status: success, failed, pending, error")
    refund: bool = Field(default=False, description="Whether to refund the wallet (only if status is failed or error)")
    send_webhook: bool = Field(default=False, description="Whether to send webhook notification")


class TransactionCountByStatus(BaseModel):
    """Transaction count grouped by status"""
    success: int = Field(default=0, description="Number of success transactions")
    failed: int = Field(default=0, description="Number of failed transactions")
    pending: int = Field(default=0, description="Number of pending transactions")
    error: int = Field(default=0, description="Number of error transactions")


class AmountByStatus(BaseModel):
    """Amount sum grouped by status"""
    success: str = Field(default="0", description="Total amount of success transactions")
    failed: str = Field(default="0", description="Total amount of failed transactions")
    pending: str = Field(default="0", description="Total amount of pending transactions")
    error: str = Field(default="0", description="Total amount of error transactions")


class ChargesByStatus(BaseModel):
    """Charges sum grouped by status"""
    success: str = Field(default="0", description="Total charges of success transactions")
    failed: str = Field(default="0", description="Total charges of failed transactions")
    pending: str = Field(default="0", description="Total charges of pending transactions")
    error: str = Field(default="0", description="Total charges of error transactions")


class PayoutStatsResponse(BaseModel):
    """Payout statistics response schema"""
    today: dict = Field(..., description="Today's statistics")
    current_month: dict = Field(..., description="Current month statistics")
    total_till_today: dict = Field(..., description="Total statistics till today")

