"""
Wallet Transaction schemas
"""

from typing import Optional
from pydantic import BaseModel, Field, ConfigDict
from uuid import UUID
from datetime import datetime
from decimal import Decimal


class WalletTransactionCreate(BaseModel):
    """Wallet Transaction creation schema"""
    user_id: UUID
    wallet_id: UUID
    txn_type: str = Field(..., description="Reason code for transaction")
    type: str = Field(..., description="debit or credit")
    amount: Decimal = Field(..., gt=0, description="Transaction amount")
    txnid: str = Field(..., description="Unique transaction ID")


class WalletTransactionResponse(BaseModel):
    """Wallet Transaction response schema"""
    id: UUID
    user_id: UUID
    wallet_id: UUID
    txn_type: str
    type: str
    amount: Decimal
    opening: Decimal
    closing: Decimal
    txnid: str
    refunded: bool
    is_refund: bool
    user_name: Optional[str] = None  # User's full name
    username: Optional[str] = None  # User's email (used as username)
    created_at: datetime
    updated_at: datetime

    model_config = ConfigDict(from_attributes=True)


class WalletTransactionCreateRequest(BaseModel):
    """Request schema for creating a wallet transaction"""
    user_id: UUID
    txn_type: str = Field(..., description="Reason code for transaction")
    type: str = Field(..., description="debit or credit")
    amount: Decimal = Field(..., gt=0, description="Transaction amount")
    txnid: str = Field(..., description="Transaction ID")
    update_main: bool = Field(default=True, description="If True, update main balance; if False, update reserve balance")


class WalletTransactionFilterRequest(BaseModel):
    """Request schema for filtering wallet transactions"""
    user_id: UUID = Field(..., description="User ID to filter transactions")
    skip: int = Field(default=0, ge=0, description="Number of records to skip")
    limit: int = Field(default=100, ge=1, le=1000, description="Maximum number of records to return")
    type: Optional[str] = Field(None, description="Filter by transaction type: debit or credit")
    txn_type: Optional[str] = Field(None, description="Filter by transaction type/reason code")
    refunded: Optional[bool] = Field(None, description="Filter by refunded status")
    is_refund: Optional[bool] = Field(None, description="Filter by is_refund status")
    txnid: Optional[str] = Field(None, description="Filter by transaction ID")


class WalletDebitRequest(BaseModel):
    """Request schema for debiting wallet"""
    user_id: UUID = Field(..., description="User ID")
    amount: Decimal = Field(..., gt=0, description="Amount to debit")


class WalletCreditRequest(BaseModel):
    """Request schema for crediting wallet"""
    user_id: UUID = Field(..., description="User ID")
    amount: Decimal = Field(..., gt=0, description="Amount to credit")


class WalletGetRequest(BaseModel):
    """Request schema for getting wallet"""
    user_id: UUID = Field(..., description="User ID")

