"""
Script to fix wallet transaction opening/closing balance chain
and update wallet balance to match the corrected chain.

Usage:
    python -m app.scripts.fix_wallet_balance_chain <user_id> [--apply]
"""

import sys
from decimal import Decimal
from sqlalchemy.orm import Session
from uuid import UUID

from app.core.database import SessionLocal
from app.models.wallet_transaction import WalletTransaction
from app.models.wallet import Wallet


def fix_wallet_balance_chain(user_id: str, apply: bool = False):
    """
    Fix wallet transaction balance chain for a user
    
    Args:
        user_id: User ID (UUID string)
        apply: If True, apply changes. If False, dry run only.
    """
    db: Session = SessionLocal()
    
    try:
        # Verify user_id is valid UUID
        try:
            user_uuid = UUID(user_id)
        except ValueError:
            print(f"❌ Error: Invalid user ID format: {user_id}")
            return False
        
        # Get wallet
        wallet = db.query(Wallet).filter(Wallet.user_id == user_uuid).first()
        if not wallet:
            print(f"❌ Error: Wallet not found for user {user_id}")
            return False
        
        print(f"✓ Found wallet for user {user_id}")
        print(f"  Current wallet balance: ₹{wallet.main:,.2f}")
        print(f"  Wallet ID: {wallet.id}")
        print()
        
        # Get all transactions in chronological order
        transactions = db.query(WalletTransaction).filter(
            WalletTransaction.user_id == user_uuid
        ).order_by(WalletTransaction.created_at.asc()).all()
        
        if not transactions:
            print("⚠️  No transactions found for this user")
            return False
        
        print(f"✓ Found {len(transactions)} transactions")
        print(f"  First transaction: {transactions[0].created_at}")
        print(f"  Last transaction: {transactions[-1].created_at}")
        print()
        
        # Recalculate balances starting from 0
        current_balance = Decimal('0.00')
        updates_needed = []
        correct_count = 0
        
        print("Recalculating transaction balances...")
        print("-" * 100)
        
        for i, txn in enumerate(transactions, 1):
            opening_balance = current_balance
            
            # Calculate closing balance based on transaction type
            if txn.type == 'credit':
                closing_balance = opening_balance + txn.amount
            else:  # debit
                closing_balance = opening_balance - txn.amount
            
            # Check if update is needed
            if txn.opening != opening_balance or txn.closing != closing_balance:
                updates_needed.append({
                    'txn': txn,
                    'old_opening': txn.opening,
                    'new_opening': opening_balance,
                    'old_closing': txn.closing,
                    'new_closing': closing_balance
                })
            else:
                correct_count += 1
            
            # Update current balance for next transaction
            current_balance = closing_balance
        
        print("-" * 100)
        print()
        print(f"Final calculated balance: ₹{current_balance:,.2f}")
        print(f"Current wallet balance: ₹{wallet.main:,.2f}")
        print(f"Difference: ₹{current_balance - wallet.main:,.2f}")
        print()
        print(f"Transactions status:")
        print(f"  ✓ Correct: {correct_count}")
        print(f"  ✗ Need update: {len(updates_needed)}")
        print()
        
        if not updates_needed and wallet.main == current_balance:
            print("✅ All transactions and wallet balance are already correct!")
            return True
        
        if not apply:
            print("=" * 100)
            print("🔍 DRY RUN MODE - No changes will be made")
            print("=" * 100)
            print()
            print("To apply changes, run with --apply flag")
            print()
            if updates_needed:
                print("Sample of transactions that would be updated:")
                for update in updates_needed[:10]:
                    txn = update['txn']
                    print(f"  {txn.txnid[:20]:20s} | Opening: ₹{update['old_opening']:>12.2f} → ₹{update['new_opening']:>12.2f} | "
                          f"Closing: ₹{update['old_closing']:>12.2f} → ₹{update['new_closing']:>12.2f}")
                if len(updates_needed) > 10:
                    print(f"  ... and {len(updates_needed) - 10} more transactions")
            if wallet.main != current_balance:
                print(f"\nWallet balance would be updated:")
                print(f"  From: ₹{wallet.main:,.2f}")
                print(f"  To:   ₹{current_balance:,.2f}")
            return False
        
        # Apply updates
        print("=" * 100)
        print("🔧 APPLYING CHANGES...")
        print("=" * 100)
        print()
        
        # Update transactions
        updated_count = 0
        for update in updates_needed:
            txn = update['txn']
            txn.opening = update['new_opening']
            txn.closing = update['new_closing']
            updated_count += 1
            if updated_count % 100 == 0:
                print(f"  Updated {updated_count}/{len(updates_needed)} transactions...")
        
        # Update wallet balance
        old_wallet_balance = wallet.main
        wallet.main = current_balance
        
        # Commit all changes
        db.commit()
        
        print()
        print("✅ Successfully updated:")
        print(f"  • {updated_count} wallet transactions (opening/closing balances)")
        if old_wallet_balance != current_balance:
            print(f"  • Wallet balance: ₹{old_wallet_balance:,.2f} → ₹{current_balance:,.2f}")
        else:
            print(f"  • Wallet balance: Already correct (₹{current_balance:,.2f})")
        print()
        print("✅ All changes committed successfully!")
        return True
        
    except Exception as e:
        db.rollback()
        print(f"❌ Error: {e}")
        import traceback
        traceback.print_exc()
        return False
    finally:
        db.close()


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python -m app.scripts.fix_wallet_balance_chain <user_id> [--apply]")
        print("  user_id: UUID of the user")
        print("  --apply: Apply changes (default is dry-run)")
        sys.exit(1)
    
    user_id = sys.argv[1]
    apply = "--apply" in sys.argv
    
    success = fix_wallet_balance_chain(user_id, apply=apply)
    sys.exit(0 if success else 1)

