"""
Script to fix wallet transaction opening/closing balance chain
and update wallet balance to match the corrected chain.

This script:
1. Reads all wallet transactions for a user in chronological order
2. Recalculates opening and closing balances starting from 0
3. Updates each transaction with correct opening/closing balances
4. Updates the wallet balance to match the final closing balance
"""

import sys
from decimal import Decimal
from sqlalchemy.orm import Session
from sqlalchemy import text
from uuid import UUID

# Add parent directory to path to import app modules
sys.path.insert(0, '/app')

from app.core.database import SessionLocal
from app.models.wallet_transaction import WalletTransaction
from app.models.wallet import Wallet


def fix_wallet_transactions(user_id: str, dry_run: bool = True):
    """
    Fix wallet transaction balance chain for a user
    
    Args:
        user_id: User ID (UUID string)
        dry_run: If True, only show what would be changed without updating
    """
    db: Session = SessionLocal()
    
    try:
        # Verify user_id is valid UUID
        try:
            user_uuid = UUID(user_id)
        except ValueError:
            print(f"Error: Invalid user ID format: {user_id}")
            return
        
        # Get wallet
        wallet = db.query(Wallet).filter(Wallet.user_id == user_uuid).first()
        if not wallet:
            print(f"Error: Wallet not found for user {user_id}")
            return
        
        print(f"Found wallet for user {user_id}")
        print(f"Current wallet balance: ₹{wallet.main}")
        print(f"Wallet ID: {wallet.id}")
        print()
        
        # Get all transactions in chronological order
        transactions = db.query(WalletTransaction).filter(
            WalletTransaction.user_id == user_uuid
        ).order_by(WalletTransaction.created_at.asc()).all()
        
        if not transactions:
            print("No transactions found for this user")
            return
        
        print(f"Found {len(transactions)} transactions")
        print(f"First transaction: {transactions[0].created_at}")
        print(f"Last transaction: {transactions[-1].created_at}")
        print()
        
        # Recalculate balances starting from 0
        current_balance = Decimal('0.00')
        updates_needed = []
        
        print("Recalculating transaction balances...")
        print("-" * 80)
        
        for i, txn in enumerate(transactions, 1):
            opening_balance = current_balance
            
            # Calculate closing balance based on transaction type
            if txn.type == 'credit':
                closing_balance = opening_balance + txn.amount
            else:  # debit
                closing_balance = opening_balance - txn.amount
            
            # Check if update is needed
            if txn.opening != opening_balance or txn.closing != closing_balance:
                updates_needed.append({
                    'txn': txn,
                    'old_opening': txn.opening,
                    'new_opening': opening_balance,
                    'old_closing': txn.closing,
                    'new_closing': closing_balance
                })
            
            # Update current balance for next transaction
            current_balance = closing_balance
            
            # Show progress for first 10 and last 10 transactions
            if i <= 10 or i > len(transactions) - 10:
                status = "✓" if txn.opening == opening_balance and txn.closing == closing_balance else "✗"
                print(f"{status} [{i:4d}] {txn.txnid[:15]:15s} | {txn.type:6s} | "
                      f"Opening: ₹{opening_balance:>12.2f} | Closing: ₹{closing_balance:>12.2f}")
                if txn.opening != opening_balance or txn.closing != closing_balance:
                    print(f"    OLD: Opening: ₹{txn.opening:>12.2f} | Closing: ₹{txn.closing:>12.2f}")
        
        print("-" * 80)
        print(f"\nFinal calculated balance: ₹{current_balance}")
        print(f"Current wallet balance: ₹{wallet.main}")
        print(f"Difference: ₹{current_balance - wallet.main}")
        print()
        
        if not updates_needed:
            print("✓ All transactions already have correct balances!")
            return
        
        print(f"Found {len(updates_needed)} transactions that need updating")
        print()
        
        if dry_run:
            print("DRY RUN MODE - No changes will be made")
            print("To apply changes, run with dry_run=False")
            print()
            print("Sample of transactions to update:")
            for update in updates_needed[:5]:
                txn = update['txn']
                print(f"  {txn.txnid}: Opening {update['old_opening']} → {update['new_opening']}, "
                      f"Closing {update['old_closing']} → {update['new_closing']}")
            if len(updates_needed) > 5:
                print(f"  ... and {len(updates_needed) - 5} more")
        else:
            print("Applying updates...")
            
            # Update transactions
            updated_count = 0
            for update in updates_needed:
                txn = update['txn']
                txn.opening = update['new_opening']
                txn.closing = update['new_closing']
                updated_count += 1
            
            # Update wallet balance
            wallet.main = current_balance
            
            # Commit all changes
            db.commit()
            
            print(f"✓ Updated {updated_count} transactions")
            print(f"✓ Updated wallet balance to ₹{current_balance}")
            print()
            print("All changes committed successfully!")
        
    except Exception as e:
        db.rollback()
        print(f"Error: {e}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python fix_wallet_transactions.py <user_id> [--apply]")
        print("  user_id: UUID of the user")
        print("  --apply: Apply changes (default is dry-run)")
        sys.exit(1)
    
    user_id = sys.argv[1]
    dry_run = "--apply" not in sys.argv
    
    if dry_run:
        print("=" * 80)
        print("DRY RUN MODE - No changes will be made")
        print("=" * 80)
        print()
    
    fix_wallet_transactions(user_id, dry_run=dry_run)

