"""
Admin service with business logic
"""

from typing import Optional
from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.admin_repository import AdminRepository
from app.schemas.admin import AdminCreate, AdminUpdate, AdminResponse
from app.core.security import get_password_hash, verify_password, create_access_token
from app.core.exceptions import ValidationError, ConflictError, UnauthorizedError
from app.models.admin import Admin


class AdminService(BaseService[Admin, AdminCreate, AdminUpdate]):
    """Admin service with business logic"""

    def __init__(self, db: Session):
        repository = AdminRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: AdminCreate, db: Session) -> dict:
        """Validate admin creation"""
        # Check if email already exists
        admin_repo = AdminRepository(db)
        existing_admin = admin_repo.get_by_email(data.email)
        if existing_admin:
            raise ConflictError("Email already registered")
        
        validated_data = data.model_dump(exclude={"password"})
        validated_data["password_hash"] = get_password_hash(data.password)
        return validated_data

    def register(self, data: AdminCreate) -> tuple[Admin, str]:
        """
        Register a new admin and return admin with access token
        
        Returns:
            Tuple of (Admin, access_token)
        """
        admin_data = self.validate_create(data, self.db)
        admin = self.repository.create(**admin_data)
        
        # Create access token
        access_token = create_access_token(
            data={"sub": str(admin.id), "role": "admin"}
        )
        
        return admin, access_token

    def login(self, email: str, password: str) -> tuple[Admin, str]:
        """
        Authenticate admin and return admin with access token
        
        Returns:
            Tuple of (Admin, access_token)
            
        Raises:
            UnauthorizedError: If credentials are invalid
        """
        admin_repo = AdminRepository(self.db)
        admin = admin_repo.get_by_email(email)
        
        if not admin or not verify_password(password, admin.password_hash):
            raise UnauthorizedError("Invalid email or password")
        
        if not admin.is_active:
            raise UnauthorizedError("Admin account is inactive")
        
        # Create access token
        access_token = create_access_token(
            data={"sub": str(admin.id), "role": "admin"}
        )
        
        return admin, access_token

