"""
API Key service with business logic
"""

import secrets
import string
from typing import Optional
from sqlalchemy.orm import Session
from app.repositories.api_key_repository import APIKeyRepository
from app.schemas.api_key import APIKeyUpdate
from app.core.exceptions import NotFoundError
from app.core.security import encrypt_api_secret
from app.models.api_key import APIKey
from uuid import UUID


class APIKeyService:
    """API Key service with business logic"""

    def __init__(self, db: Session):
        self.repository = APIKeyRepository(db)
        self.db = db

    def _generate_api_key(self) -> str:
        """Generate a random API key"""
        # Generate a 32-character alphanumeric API key
        alphabet = string.ascii_letters + string.digits
        return ''.join(secrets.choice(alphabet) for _ in range(32))

    def _generate_api_secret(self) -> str:
        """Generate a random API secret"""
        # Generate a 64-character alphanumeric API secret
        alphabet = string.ascii_letters + string.digits
        return ''.join(secrets.choice(alphabet) for _ in range(64))

    def generate_or_regenerate(self, user_id: UUID, db: Session) -> dict:
        """
        Generate or regenerate API key pair for a user
        If user already has an API key, updates it instead of creating new
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            Dictionary with api_key, api_secret (plain text, only shown once), and other fields
        """
        # Generate new key pair
        api_key = self._generate_api_key()
        api_secret_plain = self._generate_api_secret()
        api_secret_encrypted = encrypt_api_secret(api_secret_plain)  # Encrypt for storage (raw, not hashed, can be decrypted)
        
        # Check if user already has an API key
        existing = self.repository.get_by_user_id(str(user_id))
        
        if existing:
            # Update existing record
            updated = self.repository.update(
                str(existing.id),
                api_key=api_key,
                api_secret=api_secret_encrypted,  # Store encrypted (raw, not hashed)
                active=True
            )
            # Return with plain secret (only time it's visible)
            return {
                "id": updated.id,
                "user_id": updated.user_id,
                "api_key": updated.api_key,
                "api_secret": api_secret_plain,  # Plain secret, only shown once
                "active": updated.active,
                "ips": updated.ips or [],
                "created_at": updated.created_at,
                "updated_at": updated.updated_at,
            }
        else:
            # Create new record
            created = self.repository.create(
                user_id=user_id,
                api_key=api_key,
                api_secret=api_secret_encrypted,  # Store encrypted (raw, not hashed)
                active=True,
                ips=[]
            )
            # Return with plain secret (only time it's visible)
            return {
                "id": created.id,
                "user_id": created.user_id,
                "api_key": created.api_key,
                "api_secret": api_secret_plain,  # Plain secret, only shown once
                "active": created.active,
                "ips": created.ips or [],
                "created_at": created.created_at,
                "updated_at": created.updated_at,
            }

    def get_by_user_id(self, user_id: UUID, db: Session) -> APIKey:
        """
        Get API key for a user (without secret)
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            APIKey instance (without secret)
        """
        api_key = self.repository.get_by_user_id(str(user_id))
        if not api_key:
            raise NotFoundError("API key not found")
        return api_key

    def update(self, user_id: UUID, data: APIKeyUpdate, db: Session) -> APIKey:
        """
        Update API key (active status and IPs)
        
        Args:
            user_id: User ID
            data: Update data
            db: Database session
            
        Returns:
            Updated APIKey instance
        """
        api_key = self.repository.get_by_user_id(str(user_id))
        if not api_key:
            raise NotFoundError("API key not found")
        
        update_data = data.model_dump(exclude_unset=True)
        return self.repository.update(str(api_key.id), **update_data)

    def verify_api_key(self, api_key: str) -> Optional[APIKey]:
        """
        Verify API key
        
        Args:
            api_key: API key string
            
        Returns:
            APIKey instance if valid, None otherwise
        """
        api_key_obj = self.repository.get_by_api_key(api_key)
        if not api_key_obj:
            return None
        
        if not api_key_obj.active:
            return None
        
        return api_key_obj

    def add_ip_to_whitelist(self, user_id: UUID, ip: str, db: Session) -> APIKey:
        """
        Add an IP address to the whitelist
        
        Args:
            user_id: User ID
            ip: IP address to add
            db: Database session
            
        Returns:
            Updated APIKey instance
        """
        api_key = self.repository.get_by_user_id(str(user_id))
        if not api_key:
            raise NotFoundError("API key not found")
        
        # Get current IPs and create a new list (important for SQLAlchemy to detect changes)
        current_ips = list(api_key.ips) if api_key.ips else []
        
        # Add IP if not already present
        if ip not in current_ips:
            current_ips.append(ip)
            # Create a new list to ensure SQLAlchemy detects the change
            updated_ips = list(current_ips)
            return self.repository.update(str(api_key.id), ips=updated_ips)
        
        return api_key

    def get_whitelisted_ips(self, user_id: UUID, db: Session) -> list:
        """
        Get list of whitelisted IPs for user's API key
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            List of IP addresses
        """
        api_key = self.repository.get_by_user_id(str(user_id))
        if not api_key:
            raise NotFoundError("API key not found")
        
        return api_key.ips or []

    def remove_ip_from_whitelist(self, user_id: UUID, ip: str, db: Session) -> APIKey:
        """
        Remove an IP address from the whitelist
        
        Args:
            user_id: User ID
            ip: IP address to remove
            db: Database session
            
        Returns:
            Updated APIKey instance
        """
        api_key = self.repository.get_by_user_id(str(user_id))
        if not api_key:
            raise NotFoundError("API key not found")
        
        # Get current IPs and create a new list (important for SQLAlchemy to detect changes)
        current_ips = list(api_key.ips) if api_key.ips else []
        
        # Remove IP if present
        if ip in current_ips:
            current_ips.remove(ip)
            # Create a new list to ensure SQLAlchemy detects the change
            updated_ips = list(current_ips)
            return self.repository.update(str(api_key.id), ips=updated_ips)
        
        return api_key

