"""
Charge service with business logic
"""

from typing import Optional, List
from decimal import Decimal
from sqlalchemy.orm import Session
from app.repositories.charge_repository import ChargeRepository
from app.repositories.service_repository import ServiceRepository
from app.core.exceptions import ValidationError, NotFoundError
from app.models.charge import Charge
from uuid import UUID


class ChargeService:
    """Charge service with business logic"""

    def __init__(self, db: Session):
        self.charge_repo = ChargeRepository(db)
        self.service_repo = ServiceRepository(db)
        self.db = db

    def create_charge(
        self,
        service_id: UUID,
        from_amount: Decimal,
        to_amount: Decimal,
        type: str,
        is_percent: bool,
        amount: Decimal,
        user_id: Optional[UUID] = None,
        gst: Optional[Decimal] = None,
        db: Session = None
    ) -> Charge:
        """
        Create a charge slab with overlap prevention
        
        Args:
            service_id: Service ID
            from_amount: Minimum transaction amount (0 for no limit)
            to_amount: Maximum transaction amount (0 for unlimited)
            type: Charge type ("surcharge" or "commission")
            is_percent: True for percentage, False for fixed amount
            amount: Charge amount or percentage value
            user_id: User ID (None for global charges)
            gst: GST percentage (None = default 18%, 0 = no GST)
            db: Database session
            
        Returns:
            Created Charge instance
            
        Raises:
            ValidationError: If service not found, invalid type, or overlapping range
        """
        # Validate service exists
        service = self.service_repo.get_by_id(str(service_id))
        if not service:
            raise NotFoundError(f"Service with ID '{service_id}'")
        
        # Validate charge type
        if type not in ["surcharge", "commission"]:
            raise ValidationError(f"Invalid charge type: {type}. Must be 'surcharge' or 'commission'")
        
        # Validate amount range
        if from_amount < Decimal('0.00'):
            raise ValidationError("from_amount cannot be negative")
        if to_amount < Decimal('0.00'):
            raise ValidationError("to_amount cannot be negative")
        if from_amount > Decimal('0.00') and to_amount > Decimal('0.00') and from_amount > to_amount:
            raise ValidationError("from_amount cannot be greater than to_amount")
        
        # Validate amount value
        if amount <= Decimal('0.00'):
            raise ValidationError("Charge amount must be greater than 0")
        if is_percent and amount > Decimal('100.00'):
            raise ValidationError("Percentage charge cannot exceed 100%")
        
        # Validate GST if provided
        if gst is not None:
            if gst < Decimal('0.00'):
                raise ValidationError("GST cannot be negative")
            if gst > Decimal('100.00'):
                raise ValidationError("GST cannot exceed 100%")
        
        # Check for overlapping charges
        user_id_str = str(user_id) if user_id else None
        overlapping = self.charge_repo.get_overlapping_charges(
            service_id=str(service_id),
            from_amount=from_amount,
            to_amount=to_amount,
            user_id=user_id_str
        )
        
        if overlapping:
            overlap_details = []
            for charge in overlapping:
                range_desc = "all amounts" if (charge.from_amount == Decimal('0.00') and charge.to_amount == Decimal('0.00')) else f"{charge.from_amount} to {charge.to_amount if charge.to_amount > Decimal('0.00') else 'unlimited'}"
                overlap_details.append(f"ID {charge.id}: {range_desc}")
            
            raise ValidationError(
                f"Charge range overlaps with existing charges: {', '.join(overlap_details)}. "
                "Please adjust the amount range to avoid overlaps."
            )
        
        # Create charge
        charge = self.charge_repo.create(
            service_id=service_id,
            user_id=user_id,
            from_amount=from_amount,
            to_amount=to_amount,
            type=type,
            is_percent=is_percent,
            amount=amount,
            gst=gst
        )
        
        return charge

    def update_charge(
        self,
        charge_id: int,
        service_id: Optional[UUID] = None,
        user_id: Optional[UUID] = None,
        from_amount: Optional[Decimal] = None,
        to_amount: Optional[Decimal] = None,
        type: Optional[str] = None,
        is_percent: Optional[bool] = None,
        amount: Optional[Decimal] = None,
        gst: Optional[Decimal] = None,
        db: Session = None
    ) -> Charge:
        """
        Update a charge slab with overlap prevention
        
            Args:
            charge_id: Charge ID
            service_id: Service ID (optional)
            user_id: User ID (optional)
            from_amount: Minimum transaction amount (optional)
            to_amount: Maximum transaction amount (optional)
            type: Charge type (optional)
            is_percent: True for percentage, False for fixed amount (optional)
            amount: Charge amount or percentage value (optional)
            gst: GST percentage (None = default 18%, 0 = no GST) (optional)
            db: Database session
            
        Returns:
            Updated Charge instance
            
        Raises:
            NotFoundError: If charge not found
            ValidationError: If invalid data or overlapping range
        """
        # Get existing charge
        charge = self.charge_repo.get_by_id(str(charge_id))
        if not charge:
            raise NotFoundError(f"Charge with ID '{charge_id}'")
        
        # Prepare update data
        update_data = {}
        
        if service_id is not None:
            # Validate service exists
            service = self.service_repo.get_by_id(str(service_id))
            if not service:
                raise NotFoundError(f"Service with ID '{service_id}'")
            update_data["service_id"] = service_id
        
        if user_id is not None:
            update_data["user_id"] = user_id
        
        if from_amount is not None:
            if from_amount < Decimal('0.00'):
                raise ValidationError("from_amount cannot be negative")
            update_data["from_amount"] = from_amount
        
        if to_amount is not None:
            if to_amount < Decimal('0.00'):
                raise ValidationError("to_amount cannot be negative")
            update_data["to_amount"] = to_amount
        
        if from_amount is not None and to_amount is not None:
            if from_amount > Decimal('0.00') and to_amount > Decimal('0.00') and from_amount > to_amount:
                raise ValidationError("from_amount cannot be greater than to_amount")
        
        if type is not None:
            if type not in ["surcharge", "commission"]:
                raise ValidationError(f"Invalid charge type: {type}. Must be 'surcharge' or 'commission'")
            update_data["type"] = type
        
        if is_percent is not None:
            update_data["is_percent"] = is_percent
        
        if amount is not None:
            if amount <= Decimal('0.00'):
                raise ValidationError("Charge amount must be greater than 0")
            if (is_percent if is_percent is not None else charge.is_percent) and amount > Decimal('100.00'):
                raise ValidationError("Percentage charge cannot exceed 100%")
            update_data["amount"] = amount
        
        if gst is not None:
            if gst < Decimal('0.00'):
                raise ValidationError("GST cannot be negative")
            if gst > Decimal('100.00'):
                raise ValidationError("GST cannot exceed 100%")
            update_data["gst"] = gst
        
        # Check for overlaps if amount range is being updated
        if from_amount is not None or to_amount is not None:
            final_from = from_amount if from_amount is not None else charge.from_amount
            final_to = to_amount if to_amount is not None else charge.to_amount
            final_service_id = str(service_id if service_id is not None else charge.service_id)
            final_user_id = str(user_id if user_id is not None else charge.user_id) if (user_id is not None or charge.user_id) else None
            
            overlapping = self.charge_repo.get_overlapping_charges(
                service_id=final_service_id,
                from_amount=final_from,
                to_amount=final_to,
                user_id=final_user_id,
                exclude_id=charge_id
            )
            
            if overlapping:
                overlap_details = []
                for overlap_charge in overlapping:
                    range_desc = "all amounts" if (overlap_charge.from_amount == Decimal('0.00') and overlap_charge.to_amount == Decimal('0.00')) else f"{overlap_charge.from_amount} to {overlap_charge.to_amount if overlap_charge.to_amount > Decimal('0.00') else 'unlimited'}"
                    overlap_details.append(f"ID {overlap_charge.id}: {range_desc}")
                
                raise ValidationError(
                    f"Updated charge range overlaps with existing charges: {', '.join(overlap_details)}. "
                    "Please adjust the amount range to avoid overlaps."
                )
        
        # Update charge
        updated_charge = self.charge_repo.update(str(charge_id), **update_data)
        
        return updated_charge

    def get_charges(
        self,
        service_id: Optional[UUID] = None,
        user_id: Optional[UUID] = None,
        skip: int = 0,
        limit: int = 100
    ) -> List[Charge]:
        """
        Get charges with optional filtering
        
        Args:
            service_id: Service ID (optional)
            user_id: User ID (optional)
            skip: Number of records to skip
            limit: Maximum number of records to return
            
        Returns:
            List of Charge instances
        """
        if service_id:
            return self.charge_repo.get_by_service_id(str(service_id), str(user_id) if user_id else None)
        else:
            query = self.db.query(self.charge_repo.model)
            
            if user_id:
                query = query.filter(
                    or_(
                        self.charge_repo.model.user_id == user_id,
                        self.charge_repo.model.user_id.is_(None)
                    )
                )
            
            return query.order_by(
                self.charge_repo.model.service_id,
                self.charge_repo.model.from_amount.asc()
            ).offset(skip).limit(limit).all()

    def calculate_charge(
        self,
        service_id: UUID,
        transaction_amount: Decimal,
        user_id: Optional[UUID] = None,
        db: Session = None
    ) -> Decimal:
        """
        Calculate charge for a transaction based on amount and applicable charge slabs
        
        Args:
            service_id: Service ID
            transaction_amount: Transaction amount
            user_id: User ID (optional, for user-specific charges)
            
        Returns:
            Calculated charge amount
        """
        # Get applicable charges for this service and user
        user_id_str = str(user_id) if user_id else None
        charges = self.charge_repo.get_by_service_id(str(service_id), user_id_str)
        
        # Find applicable charge slab
        applicable_charge = None
        for charge in charges:
            # Universal charge (from=0, to=0 applies to all)
            if charge.from_amount == Decimal('0.00') and charge.to_amount == Decimal('0.00'):
                applicable_charge = charge
                break
            
            # Range-based charge
            if charge.from_amount <= transaction_amount:
                if charge.to_amount == Decimal('0.00') or transaction_amount <= charge.to_amount:
                    applicable_charge = charge
                    break
        
        if not applicable_charge:
            return Decimal('0.00')
        
        # Calculate charge
        if applicable_charge.is_percent:
            charge_amount = (transaction_amount * applicable_charge.amount) / Decimal('100.00')
        else:
            charge_amount = applicable_charge.amount
        
        return charge_amount.quantize(Decimal('0.01'))
    
    def calculate_charge_and_gst(
        self,
        service_id: UUID,
        transaction_amount: Decimal,
        user_id: Optional[UUID] = None,
        db: Session = None
    ) -> tuple[Decimal, Decimal]:
        """
        Calculate charge and GST for a transaction based on amount and applicable charge slabs
        
        Args:
            service_id: Service ID
            transaction_amount: Transaction amount
            user_id: User ID (optional, for user-specific charges)
            
        Returns:
            Tuple of (charge_amount, gst_amount)
            - charge_amount: Calculated charge amount
            - gst_amount: Calculated GST amount on charge (default 18% if GST is null, 0% if GST is 0)
        """
        # Get applicable charges for this service and user
        user_id_str = str(user_id) if user_id else None
        charges = self.charge_repo.get_by_service_id(str(service_id), user_id_str)
        
        # Find applicable charge slab
        applicable_charge = None
        for charge in charges:
            # Universal charge (from=0, to=0 applies to all)
            if charge.from_amount == Decimal('0.00') and charge.to_amount == Decimal('0.00'):
                applicable_charge = charge
                break
            
            # Range-based charge
            if charge.from_amount <= transaction_amount:
                if charge.to_amount == Decimal('0.00') or transaction_amount <= charge.to_amount:
                    applicable_charge = charge
                    break
        
        if not applicable_charge:
            return Decimal('0.00'), Decimal('0.00')
        
        # Calculate charge
        if applicable_charge.is_percent:
            charge_amount = (transaction_amount * applicable_charge.amount) / Decimal('100.00')
        else:
            charge_amount = applicable_charge.amount
        
        charge_amount = charge_amount.quantize(Decimal('0.01'))
        
        # Calculate GST on charge
        # If GST is null, default to 18%
        # If GST is 0, no GST
        # Otherwise use the configured GST percentage
        if applicable_charge.gst is None:
            gst_percentage = Decimal('18.00')  # Default 18%
        elif applicable_charge.gst == Decimal('0.00'):
            gst_percentage = Decimal('0.00')  # No GST
        else:
            gst_percentage = applicable_charge.gst
        
        gst_amount = (charge_amount * gst_percentage) / Decimal('100.00')
        gst_amount = gst_amount.quantize(Decimal('0.01'))
        
        return charge_amount, gst_amount

