"""
Kafka Producer Service for high-throughput message publishing
"""

import json
import logging
from typing import Dict, Any, Optional
from datetime import datetime
from confluent_kafka import Producer
from confluent_kafka.admin import AdminClient, NewTopic
from confluent_kafka.cimpl import KafkaException
from app.core.config import settings

logger = logging.getLogger(__name__)


class KafkaProducerService:
    """Kafka producer service for publishing messages with high throughput"""
    
    def __init__(self):
        """Initialize Kafka producer with optimized settings for high throughput"""
        self.producer_config = {
            'bootstrap.servers': settings.KAFKA_BOOTSTRAP_SERVERS,
            'batch.size': settings.KAFKA_BATCH_SIZE,
            'linger.ms': settings.KAFKA_LINGER_MS,
            'compression.type': 'snappy',  # Compression for better throughput
            'acks': 'all',  # Must be 'all' when idempotence is enabled
            'retries': 3,
            'max.in.flight.requests.per.connection': 5,
            'enable.idempotence': True,  # Prevent duplicate messages
            'queue.buffering.max.messages': 100000,  # Large buffer for high throughput
            'queue.buffering.max.kbytes': 1048576,  # 1GB buffer
        }
        self.producer = Producer(self.producer_config)
        self._ensure_topic_exists()
    
    def _ensure_topic_exists(self):
        """Ensure Kafka topic exists with proper configuration for high throughput"""
        admin_client = AdminClient({'bootstrap.servers': settings.KAFKA_BOOTSTRAP_SERVERS})
        
        topic_list = [NewTopic(
            settings.KAFKA_PAYIN_CREDIT_TOPIC,
            num_partitions=10,  # Multiple partitions for parallel processing
            replication_factor=1,  # Adjust based on your Kafka cluster setup
            config={
                'retention.ms': 604800000,  # 7 days retention
                'compression.type': 'snappy',
            }
        )]
        
        try:
            futures = admin_client.create_topics(topic_list)
            for topic, future in futures.items():
                try:
                    future.result()  # Wait for topic creation
                    logger.info(f"Topic {topic} created successfully")
                except KafkaException as e:
                    if e.args[0].code() == KafkaException.TOPIC_ALREADY_EXISTS:
                        logger.info(f"Topic {topic} already exists")
                    else:
                        logger.error(f"Failed to create topic {topic}: {e}")
        except Exception as e:
            logger.error(f"Error ensuring topic exists: {e}")
    
    def _delivery_callback(self, err, msg):
        """Callback for message delivery confirmation"""
        if err:
            logger.error(f"Message delivery failed: {err}")
        else:
            logger.debug(f"Message delivered to {msg.topic()} [{msg.partition()}] at offset {msg.offset()}")
    
    def publish_payin_credit_event(
        self,
        payin_txn_id: int,
        user_id: str,
        txnid: str,
        amount: str,
        timestamp: Optional[datetime] = None
    ):
        """
        Publish payin credit event to Kafka
        
        Args:
            payin_txn_id: Payin transaction ID
            user_id: User ID (used for partitioning)
            txnid: Transaction ID
            amount: Amount to credit
            timestamp: Event timestamp (defaults to now)
        """
        if timestamp is None:
            timestamp = datetime.utcnow()
        
        # Calculate target timestamp (current + delay)
        target_timestamp = timestamp.timestamp() + settings.PAYIN_CREDIT_DELAY_SECONDS
        
        event = {
            "payin_txn_id": payin_txn_id,
            "user_id": user_id,
            "txnid": txnid,
            "amount": amount,
            "created_at": timestamp.isoformat(),
            "target_timestamp": target_timestamp,  # When this should be processed
            "delay_seconds": settings.PAYIN_CREDIT_DELAY_SECONDS
        }
        
        try:
            # Use user_id for partitioning to ensure same user's transactions are ordered
            # JSON serialize the event
            message_value = json.dumps(event).encode('utf-8')
            
            # Publish to Kafka with user_id as partition key for better distribution
            self.producer.produce(
                settings.KAFKA_PAYIN_CREDIT_TOPIC,
                key=user_id.encode('utf-8'),  # Partition by user_id
                value=message_value,
                callback=self._delivery_callback,
                timestamp=int(timestamp.timestamp() * 1000)  # Kafka timestamp in milliseconds
            )
            
            # Trigger delivery callbacks (non-blocking)
            self.producer.poll(0)
            
            logger.info(f"Published payin credit event for txnid {txnid} to Kafka")
            
        except Exception as e:
            logger.error(f"Failed to publish payin credit event: {e}", exc_info=True)
            raise
    
    def flush(self, timeout: float = 10.0):
        """Flush all pending messages"""
        self.producer.flush(timeout)
    
    def close(self):
        """Close the producer"""
        self.flush()
        self.producer = None


# Global producer instance
_kafka_producer: Optional[KafkaProducerService] = None


def get_kafka_producer() -> KafkaProducerService:
    """Get or create global Kafka producer instance"""
    global _kafka_producer
    if _kafka_producer is None:
        _kafka_producer = KafkaProducerService()
    return _kafka_producer

