"""
KYC Document service
"""

from typing import List
from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.kyc_doc_repository import KYCDocRepository
from app.schemas.kyc_doc import KYCDocCreate, KYCDocUpdate
from app.core.exceptions import ValidationError
from app.common.enums import KYCDocStatus
from app.models.kyc_doc import KYCDoc
from uuid import UUID


class KYCDocService(BaseService[KYCDoc, KYCDocCreate, KYCDocUpdate]):
    """KYC Document service"""

    def __init__(self, db: Session):
        repository = KYCDocRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: KYCDocCreate, db: Session) -> dict:
        """Validate KYC document creation"""
        validated_data = data.model_dump(exclude_unset=True)
        # Set default status to submitted
        if "status" not in validated_data:
            validated_data["status"] = KYCDocStatus.SUBMITTED.value
        return validated_data

    def validate_update(self, data: KYCDocUpdate, db: Session) -> dict:
        """Validate KYC document update"""
        validated_data = data.model_dump(exclude_unset=True)
        # Validate status if provided
        if "status" in validated_data:
            status = validated_data["status"]
            if status not in [s.value for s in KYCDocStatus]:
                raise ValidationError(f"Invalid status: {status}. Valid statuses are: {', '.join([s.value for s in KYCDocStatus])}")
        return validated_data

    def create(self, data: KYCDocCreate, db: Session, user_id: UUID) -> KYCDoc:
        """
        Create KYC document for a user
        
        Args:
            data: KYC Document creation data
            db: Database session
            user_id: User ID
            
        Returns:
            Created KYCDoc instance
        """
        validated_data = self.validate_create(data, db)
        validated_data["user_id"] = user_id
        return self.repository.create(**validated_data)

    def get_by_kyc_id(self, kyc_id: UUID, db: Session) -> List[KYCDoc]:
        """
        Get all KYC documents for a specific KYC
        
        Args:
            kyc_id: KYC ID
            db: Database session
            
        Returns:
            List of KYCDoc instances
        """
        return self.repository.get_by_kyc_id(kyc_id)

