"""
KYC service with business logic
"""

from typing import Dict, Any
from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.kyc_repository import KYCRepository
from app.schemas.kyc import KYCCreate, KYCUpdate
from app.core.exceptions import ValidationError, ConflictError, NotFoundError
from app.common.enums import KYCStatus
from app.models.kyc import KYC
from uuid import UUID


class KYCService(BaseService[KYC, KYCCreate, KYCUpdate]):
    """KYC service with business logic"""

    def __init__(self, db: Session):
        repository = KYCRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: KYCCreate, db: Session) -> dict:
        """Validate KYC creation"""
        from app.common.kyc_config import is_valid_ownership_type
        
        # Validate ownership type
        if not is_valid_ownership_type(data.ownership_type):
            raise ValidationError(f"Invalid ownership type: {data.ownership_type}")
        
        validated_data = data.model_dump()
        # Set default status
        validated_data["status"] = KYCStatus.PENDING.value
        return validated_data

    def validate_update(self, data: KYCUpdate, db: Session) -> dict:
        """Validate KYC update with status transition validation"""
        validated_data = data.model_dump(exclude_unset=True)
        
        # Validate status transition if status is being updated
        if "status" in validated_data:
            new_status = validated_data["status"]
            if new_status not in [s.value for s in KYCStatus]:
                raise ValidationError(f"Invalid status: {new_status}")
        
        return validated_data

    def create(self, data: KYCCreate, db: Session, user_id: UUID) -> KYC:
        """
        Create or update KYC for a user (user can only have one KYC)
        If user already has a KYC, it will be updated instead of creating a new one
        
        Args:
            data: KYC creation data
            db: Database session
            user_id: User ID
            
        Returns:
            Created or updated KYC instance
        """
        validated_data = self.validate_create(data, db)
        validated_data["user_id"] = user_id
        
        # Check if user already has a KYC
        existing_user_kyc = self.repository.get_latest_by_user_id(user_id)
        
        if existing_user_kyc:
            # Update existing KYC instead of creating new one
            # Only allow update if status is rejected or pending (not under review or approved)
            if existing_user_kyc.status == KYCStatus.UNDER_REVIEW.value:
                raise ConflictError("Cannot edit KYC while it is under review")
            if existing_user_kyc.status == KYCStatus.APPROVED.value:
                raise ConflictError("Cannot edit approved KYC")
            if existing_user_kyc.status not in [KYCStatus.REJECTED.value, KYCStatus.PENDING.value]:
                raise ConflictError("User already has a KYC submission that cannot be edited")
            
            # Update the existing KYC and reset status to pending
            validated_data["status"] = KYCStatus.PENDING.value
            return self.repository.update(str(existing_user_kyc.id), **validated_data)
        
        # Create new KYC
        return self.repository.create(**validated_data)

    def update_status(self, id: str, status: str, db: Session) -> KYC:
        """
        Update KYC status with validation
        
        If approving KYC, validates that all submitted required documents are approved.
        
        Args:
            id: KYC ID
            status: New status
            db: Database session
            
        Returns:
            Updated KYC instance
        """
        if status not in [s.value for s in KYCStatus]:
            raise ValidationError(f"Invalid status: {status}")
        
        # If approving, validate that all submitted required documents are approved
        if status == KYCStatus.APPROVED.value:
            kyc = self.repository.get_by_id(id)
            if not kyc:
                raise NotFoundError("KYC not found")
            
            from app.common.kyc_config import get_required_documents
            from app.repositories.kyc_doc_repository import KYCDocRepository
            from app.common.enums import KYCDocStatus
            
            # Get required documents for this ownership type
            ownership_type_str = kyc.ownership_type
            required_doc_types = get_required_documents(ownership_type_str)
            
            # Get all submitted documents for this user
            doc_repo = KYCDocRepository(db)
            user_docs = doc_repo.get_by_user_id(kyc.user_id)
            
            # Create a map of document_type -> latest document
            doc_map = {}
            for doc in sorted(user_docs, key=lambda x: x.created_at, reverse=True):
                if doc.document_type not in doc_map:
                    doc_map[doc.document_type] = doc
            
            # Check that all submitted required documents are approved
            unapproved_docs = []
            for doc_type in required_doc_types:
                doc = doc_map.get(doc_type)
                if doc:  # Document is submitted
                    if doc.status != KYCDocStatus.APPROVED.value:
                        unapproved_docs.append(doc_type)
            
            if unapproved_docs:
                raise ValidationError(
                    f"Cannot approve KYC. The following required documents are not approved: {', '.join(unapproved_docs)}"
                )
        
        return self.repository.update_status(id, status)

    def get_user_kyc_status(self, user_id: UUID, db: Session) -> dict:
        """
        Get optimized KYC status for a user (user has only one KYC)
        
        Returns:
            Dictionary with status and optional kyc
            Status values: not_submitted, pending, under_review, completed, rejected
        """
        from app.common.enums import KYCStatus
        
        user_kyc = self.repository.get_latest_by_user_id(user_id)
        
        if not user_kyc:
            return {
                "status": "not_submitted",
                "kyc": None
            }
        
        # Map database status to user-friendly status
        db_status = user_kyc.status
        
        if db_status == KYCStatus.APPROVED.value:
            status = "completed"
        elif db_status == KYCStatus.REJECTED.value:
            status = "rejected"
        elif db_status == KYCStatus.UNDER_REVIEW.value:
            status = "under_review"
        elif db_status == KYCStatus.PENDING.value:
            status = "pending"
        else:
            status = "pending"  # Default fallback
        
        return {
            "status": status,
            "kyc": user_kyc
        }

    def get_user_required_documents(self, user_id: UUID, db: Session) -> Dict[str, Any]:
        """
        Get required documents for user's KYC with their submission status
        
        Returns:
            Dictionary with ownership type, required and optional documents with status
        """
        from app.common.kyc_config import (
            get_ownership_type,
            get_required_documents,
            get_optional_documents,
            get_document_description
        )
        from app.repositories.kyc_doc_repository import KYCDocRepository
        from app.common.enums import KYCDocStatus
        
        # Get user's KYC
        user_kyc = self.repository.get_latest_by_user_id(user_id)
        if not user_kyc:
            raise NotFoundError("KYC not found. Please submit KYC first.")
        
        ownership_type_str = user_kyc.ownership_type
        ownership_config = get_ownership_type(ownership_type_str)
        if not ownership_config:
            raise ValidationError(f"Invalid ownership type: {ownership_type_str}")
        
        # Get all required and optional documents
        required_doc_types = get_required_documents(ownership_type_str)
        optional_doc_types = get_optional_documents(ownership_type_str)
        
        # Get all submitted documents for this user
        doc_repo = KYCDocRepository(db)
        user_docs = doc_repo.get_by_user_id(user_id)
        
        # Create a map of document_type -> latest document for quick lookup
        # If multiple documents of same type exist, use the latest one
        doc_map = {}
        for doc in sorted(user_docs, key=lambda x: x.created_at, reverse=True):
            if doc.document_type not in doc_map:
                doc_map[doc.document_type] = doc
        
        # Build required documents list with status
        required_docs = []
        for doc_type in required_doc_types:
            doc = doc_map.get(doc_type)
            if doc:
                status = doc.status if doc.status else KYCDocStatus.SUBMITTED.value
                required_docs.append({
                    "document_type": doc_type,
                    "document_name": doc_type.replace("_", " ").title(),
                    "description": get_document_description(doc_type),
                    "is_required": True,
                    "status": status,
                    "document_number": doc.document_number,
                    "doc_id": doc.id,
                    "submitted_at": doc.created_at
                })
            else:
                required_docs.append({
                    "document_type": doc_type,
                    "document_name": doc_type.replace("_", " ").title(),
                    "description": get_document_description(doc_type),
                    "is_required": True,
                    "status": "not_submitted",
                    "document_number": None,
                    "doc_id": None,
                    "submitted_at": None
                })
        
        # Build optional documents list with status
        optional_docs = []
        for doc_type in optional_doc_types:
            doc = doc_map.get(doc_type)
            if doc:
                status = doc.status if doc.status else KYCDocStatus.SUBMITTED.value
                optional_docs.append({
                    "document_type": doc_type,
                    "document_name": doc_type.replace("_", " ").title(),
                    "description": get_document_description(doc_type),
                    "is_required": False,
                    "status": status,
                    "document_number": doc.document_number,
                    "doc_id": doc.id,
                    "submitted_at": doc.created_at
                })
            else:
                optional_docs.append({
                    "document_type": doc_type,
                    "document_name": doc_type.replace("_", " ").title(),
                    "description": get_document_description(doc_type),
                    "is_required": False,
                    "status": "not_submitted",
                    "document_number": None,
                    "doc_id": None,
                    "submitted_at": None
                })
        
        return {
            "ownership_type": ownership_type_str,
            "ownership_type_name": ownership_config.get("name", ownership_type_str),
            "required_documents": required_docs,
            "optional_documents": optional_docs
        }

    def submit_for_review(self, user_id: UUID, db: Session) -> KYC:
        """
        Submit KYC for review
        
        Validates that:
        - KYC details are submitted
        - All required documents are submitted
        - No documents are rejected
        - KYC is not already rejected
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            Updated KYC instance with status set to UNDER_REVIEW
        """
        from app.common.kyc_config import get_required_documents
        from app.repositories.kyc_doc_repository import KYCDocRepository
        from app.common.enums import KYCDocStatus
        
        # Get user's KYC
        user_kyc = self.repository.get_latest_by_user_id(user_id)
        if not user_kyc:
            raise NotFoundError("KYC not found. Please submit KYC details first.")
        
        # Check if KYC is already rejected
        if user_kyc.status == KYCStatus.REJECTED.value:
            raise ValidationError("Cannot submit rejected KYC for review. Please resubmit with updated details.")
        
        # Check if KYC is already under review or approved
        if user_kyc.status == KYCStatus.UNDER_REVIEW.value:
            raise ConflictError("KYC is already under review")
        if user_kyc.status == KYCStatus.APPROVED.value:
            raise ConflictError("KYC is already approved")
        
        # Get required documents for this ownership type
        ownership_type_str = user_kyc.ownership_type
        required_doc_types = get_required_documents(ownership_type_str)
        
        # Get all submitted documents for this user
        doc_repo = KYCDocRepository(db)
        user_docs = doc_repo.get_by_user_id(user_id)
        
        # Create a map of document_type -> latest document
        doc_map = {}
        for doc in sorted(user_docs, key=lambda x: x.created_at, reverse=True):
            if doc.document_type not in doc_map:
                doc_map[doc.document_type] = doc
        
        # Check if all required documents are submitted
        missing_docs = []
        rejected_docs = []
        
        for doc_type in required_doc_types:
            doc = doc_map.get(doc_type)
            if not doc:
                missing_docs.append(doc_type)
            elif doc.status == KYCDocStatus.REJECTED.value:
                rejected_docs.append(doc_type)
        
        # Check for any rejected documents (including optional ones)
        for doc_type, doc in doc_map.items():
            if doc.status == KYCDocStatus.REJECTED.value and doc_type not in rejected_docs:
                rejected_docs.append(doc_type)
        
        # Build error messages
        errors = []
        if missing_docs:
            errors.append(f"Missing required documents: {', '.join(missing_docs)}")
        if rejected_docs:
            errors.append(f"Rejected documents must be resubmitted: {', '.join(rejected_docs)}")
        
        if errors:
            raise ValidationError("Cannot submit KYC for review. " + " ".join(errors))
        
        # Update KYC status to UNDER_REVIEW
        return self.repository.update_status(str(user_kyc.id), KYCStatus.UNDER_REVIEW.value)

