"""
Simple database-based scheduler for processing delayed payin wallet credits
Optimized for moderate throughput (10,000 transactions/hour)
"""

import logging
import time
from datetime import datetime
from decimal import Decimal
from sqlalchemy.orm import Session
from app.core.database import SessionLocal
from app.repositories.payin_transaction_repository import PayinTransactionRepository
from app.services.wallet_service import WalletService
from app.common.enums import TransactionType
from app.core.config import settings

logger = logging.getLogger(__name__)


class PayinCreditScheduler:
    """Simple scheduler that processes payin credits based on scheduled_credit_at timestamp"""
    
    def __init__(self):
        self.running = False
        self.poll_interval = 5  # Check every 5 seconds
    
    def process_ready_credits(self, db: Session) -> int:
        """
        Process all payin transactions that are ready to be credited
        
        Returns:
            Number of transactions processed
        """
        try:
            payin_repo = PayinTransactionRepository(db)
            now = datetime.utcnow()
            
            # Find all transactions that are ready to be credited
            # Status must be 'success', not settled, not refunded, and scheduled_credit_at <= now
            # Use raw SQL to avoid model loading issues if column doesn't exist yet
            from sqlalchemy import text
            ready_transactions = db.execute(
                text("""
                    SELECT id FROM payin_transactions 
                    WHERE status = 'success' 
                    AND settled = false 
                    AND refunded = false 
                    AND scheduled_credit_at IS NOT NULL 
                    AND scheduled_credit_at <= :now
                    LIMIT 100
                """),
                {"now": now}
            ).fetchall()
            
            # Get full transaction objects by ID
            transaction_ids = [row[0] for row in ready_transactions]
            if not transaction_ids:
                return 0
            
            # Get transaction objects directly by ID (model doesn't have scheduled_credit_at yet)
            ready_transactions = []
            for txn_id in transaction_ids:
                txn = payin_repo.get_by_id(str(txn_id))
                if txn:
                    ready_transactions.append(txn)
            
            processed_count = 0
            
            for payin_txn in ready_transactions:
                try:
                    # Double-check conditions (in case of race conditions)
                    if payin_txn.settled or payin_txn.refunded or payin_txn.status != 'success':
                        logger.debug(f"Skipping txnid {payin_txn.txnid} - already settled/refunded or status changed")
                        continue
                    
                    # Credit the wallet
                    # IMPORTANT: Credit net amount (amount - charge), not full amount
                    from decimal import Decimal

                    gross_amount = Decimal(payin_txn.amount)
                    charge_amount = Decimal(payin_txn.charge or 0)
                    net_amount = gross_amount - charge_amount

                    if net_amount <= 0:
                        logger.error(
                            f"Computed net_amount <= 0 for scheduled payin txnid {payin_txn.txnid}: "
                            f"gross={gross_amount}, charge={charge_amount}"
                        )
                        continue

                    wallet_service = WalletService(db)
                    wallet_service.process_transaction(
                        user_id=payin_txn.user_id,
                        txn_type="payin",
                        transaction_type=TransactionType.CREDIT.value,
                        amount=net_amount,
                        txnid=payin_txn.txnid,
                        db=db,
                        update_main=True
                    )
                    
                    # Mark as settled
                    payin_repo.update(
                        str(payin_txn.id),
                        settled=True
                    )
                    
                    processed_count += 1
                    logger.info(f"Successfully credited wallet for payin txnid {payin_txn.txnid}, amount: {payin_txn.amount}, settled: true")
                    
                except Exception as e:
                    logger.error(f"Error processing payin credit for txnid {payin_txn.txnid}: {e}", exc_info=True)
                    # Continue processing other transactions
            
            if processed_count > 0:
                db.commit()
                logger.info(f"Processed {processed_count} payin credit(s)")
            
            return processed_count
            
        except Exception as e:
            logger.error(f"Error in process_ready_credits: {e}", exc_info=True)
            db.rollback()
            return 0
    
    def run(self):
        """Run the scheduler loop"""
        self.running = True
        logger.info(f"Starting payin credit scheduler (checking every {self.poll_interval}s)")
        
        while self.running:
            db = None
            try:
                db = SessionLocal()
                self.process_ready_credits(db)
            except Exception as e:
                logger.error(f"Unhandled exception in scheduler loop: {e}", exc_info=True)
            finally:
                if db:
                    db.close()
            
            # Sleep before next check
            time.sleep(self.poll_interval)
    
    def stop(self):
        """Stop the scheduler"""
        self.running = False
        logger.info("Stopping payin credit scheduler")


def main():
    """Main entry point for the scheduler"""
    scheduler = PayinCreditScheduler()
    try:
        scheduler.run()
    except KeyboardInterrupt:
        logger.info("Scheduler interrupted by user")
        scheduler.stop()


if __name__ == "__main__":
    main()

