"""
Payin service with business logic
"""

import secrets
import string
from typing import Optional, Dict, Any
from decimal import Decimal
from sqlalchemy.orm import Session
from app.repositories.payin_transaction_repository import PayinTransactionRepository
from app.repositories.service_repository import ServiceRepository
from app.repositories.user_service_repository import UserServiceRepository
from app.services.wallet_service import WalletService
from app.services.providers.provider_factory import ProviderFactory
from app.core.exceptions import ValidationError, NotFoundError
from app.common.enums import UserServiceStatus, TransactionType
from app.models.payin_transaction import PayinTransaction
from uuid import UUID
from app.services.webhook_sender_service import WebhookSenderService
from app.core.config import settings
import logging

logger = logging.getLogger(__name__)

class PayinService:
    """Payin service with business logic"""

    def __init__(self, db: Session):
        self.payin_repo = PayinTransactionRepository(db)
        self.service_repo = ServiceRepository(db)
        self.user_service_repo = UserServiceRepository(db)
        self.wallet_service = WalletService(db)
        self.db = db

    def check_service_status(self, user_id: UUID, service_code: str, db: Session) -> bool:
        """
        Check if user has active service for the given service code
        
        Args:
            user_id: User ID
            service_code: Service code (e.g., "payin")
            db: Database session
            
        Returns:
            True if user has active service, False otherwise
            
        Raises:
            NotFoundError: If service not found
            ValidationError: If service is not active for user or service itself is not active
        """
        from app.common.enums import ServiceStatus
        
        # Get service by code
        service = self.service_repo.get_by_code(service_code)
        if not service:
            raise NotFoundError(f"Service with code '{service_code}' not found")
        
        # Check if the service itself is active (in services table)
        if service.status != ServiceStatus.ACTIVE.value:
            raise ValidationError(f"Service '{service_code}' is not active. Service status: {service.status}")
        
        # Check user service status
        user_service = self.user_service_repo.get_by_user_and_service(str(user_id), str(service.id))
        if not user_service:
            raise ValidationError(f"Service '{service_code}' is not assigned to user")
        
        if user_service.status != UserServiceStatus.ACTIVE.value:
            raise ValidationError(f"Service '{service_code}' is not active for user")
        
        return True

    def initiate_payin(
        self,
        user_id: UUID,
        amount: Decimal,
        charge: Decimal,
        name: str,
        email: str,
        mobile: str,
        vpa: str = None,
        user_reference_id: str = None,
        provider_name: str = None,
        db: Session = None
    ) -> PayinTransaction:
        """
        Initiate a payin transaction (create payment link/QR)
        
        Steps:
        1. Check service status for payin
        2. Generate transaction ID
        3. Call rupeeflow API via provider
        4. Create payin transaction record with status "pending" (NO wallet credit yet)
        5. Store payment URL and QR text from API response
        
        Args:
            user_id: User ID
            amount: Payin amount
            charge: Service charge
            name: Payer name
            email: Payer email
            mobile: Payer mobile number
            vpa: Payer VPA (optional)
            user_reference_id: User-provided reference ID
            db: Database session
            
        Returns:
            Created PayinTransaction instance
        """
        # Step 1: Check service status for payin
        self.check_service_status(user_id, "payin", db)
        
        # Step 2: Get provider (defaults to rupeeflow for now)
        # TODO: In future, implement user provider preference system
        provider_name = provider_name or "rupeeflow"
        
        # Step 3: Generate transaction ID
        # Format: PIN + random alphanumeric string (12 characters)
        random_string = ''.join(secrets.choice(string.ascii_uppercase + string.digits) for _ in range(12))
        txnid = f"PIN{random_string}"
        
        # Step 4: Check if user_reference_id already exists
        if user_reference_id:
            existing_txn = self.payin_repo.get_by_user_reference_id(user_reference_id)
            if existing_txn:
                raise ValidationError(f"Reference ID '{user_reference_id}' already exists. Please use a unique reference ID.")
        
        # Step 5: Get provider instance from factory
        provider = ProviderFactory.get_payin_provider(provider_name)
        payin_response = provider.create_payin(
            txnid=txnid,
            amount=amount,
            name=name,
            email=email,
            mobile=mobile,
            vpa=vpa,
            user_reference_id=user_reference_id
        )
        
        # Step 6: Create payin transaction record with status "pending"
        # NO wallet credit at this stage - only when webhook confirms payment
        api_response = payin_response.to_dict()
        
        # Map API response status to transaction status
        api_status = payin_response.status
        if api_status == "success":
            # Even if API returns success, transaction status is "pending" until webhook confirms payment
            txn_status = "pending"
        elif api_status == "failed":
            txn_status = "failed"
        elif api_status == "error":
            txn_status = "error"
        else:
            txn_status = "pending"
        
        provider_reference_id = payin_response.reference_id
        
        # Create payin transaction 
        payin_txn = self.payin_repo.create(
            user_id=user_id,
            txnid=txnid,
            user_reference_id=user_reference_id,
            amount=amount,
            charge=charge,
            payee_vpa=vpa,
            payee_name=name,
            status=txn_status,
            api_response=api_response,
            qr_text=payin_response.qr_text,
            payment_url=payin_response.payment_url,
            api_provider=provider_name,
            provider_reference_id=provider_reference_id
        )
        
        return payin_txn

    def get_payin_status(self, user_reference_id: str, user_id: UUID, db: Session) -> PayinTransaction:
        """
        Get payin transaction status by user reference ID
        
        Args:
            user_reference_id: User-provided reference ID
            user_id: User ID
            db: Database session
            
        Returns:
            PayinTransaction instance
            
        Raises:
            NotFoundError: If transaction not found
        """
        payin_txn = self.payin_repo.get_by_user_reference_id(user_reference_id)
        if not payin_txn:
            raise NotFoundError(f"Payin transaction with reference ID '{user_reference_id}' not found")
        
        # Verify transaction belongs to user
        if str(payin_txn.user_id) != str(user_id):
            raise NotFoundError(f"Payin transaction with reference ID '{user_reference_id}' not found")
        
        return payin_txn

    def get_transactions(
        self,
        user_id: Optional[UUID] = None,
        db: Session = None,
        skip: int = 0,
        limit: int = 100,
        username: Optional[str] = None,
        status: Optional[str] = None,
        txnid: Optional[str] = None,
        user_reference_id: Optional[str] = None
    ) -> list[PayinTransaction]:
        """
        Get payin transaction history with optional filtering
        
        Args:
            user_id: User ID (optional, if None returns all transactions)
            db: Database session
            skip: Number of records to skip
            limit: Maximum number of records to return
            username: Username (email) to filter by (optional)
            status: Transaction status to filter by (optional)
            txnid: Transaction ID to filter by (optional)
            user_reference_id: User reference ID to filter by (optional)
            
        Returns:
            List of PayinTransaction instances
        """
        from app.models.user import User
        
        # Build query with all filters applied BEFORE pagination
        query = self.db.query(self.payin_repo.model)
        
        # Apply user_id filter
        if user_id:
            query = query.filter(self.payin_repo.model.user_id == user_id)
        
        # Apply username filter (requires join)
        if username:
            query = query.join(User, self.payin_repo.model.user_id == User.id).filter(
                User.email.ilike(f"%{username}%")
            )
        
        # Apply status filter
        if status:
            query = query.filter(self.payin_repo.model.status.ilike(f"%{status}%"))
        
        # Apply txnid filter
        if txnid:
            query = query.filter(self.payin_repo.model.txnid == txnid)
        
        # Apply user_reference_id filter
        if user_reference_id:
            query = query.filter(self.payin_repo.model.user_reference_id == user_reference_id)
        
        # Apply ordering and pagination AFTER all filters
        transactions = query.order_by(
            self.payin_repo.model.created_at.desc()
        ).offset(skip).limit(limit).all()
        
        return transactions
    
    def handle_webhook(
        self,
        order_id: str,
        status: str,
        amount: Decimal = None,
        rrn: str = None,
        db: Session = None
    ) -> PayinTransaction:
        """
        Handle webhook callback from payment provider
        
        When webhook confirms payment (status=SUCCESS), credit the wallet.
        
        Args:
            order_id: Provider's order ID (stored as provider_reference_id)
            status: Status from provider (SUCCESS, FAILED, PENDING)
            amount: Transaction amount (for validation)
            rrn: Reference number (UTR)
            db: Database session
            
        Returns:
            Updated PayinTransaction instance
            
        Raises:
            NotFoundError: If transaction not found
        """
        # Find transaction by provider reference ID (orderId)
        payin_txn = self.payin_repo.get_by_provider_reference_id(order_id)
        if not payin_txn:
            raise NotFoundError(f"Payin transaction with order ID '{order_id}' not found")
        
        # Map provider status to our status
        status_upper = status.upper()
        if status_upper == "SUCCESS":
            txn_status = "success"
        elif status_upper == "FAILED":
            txn_status = "failed"
        elif status_upper == "PENDING":
            txn_status = "pending"
        else:
            txn_status = "pending"
        
        # Update api_response to include webhook data
        current_api_response = payin_txn.api_response or {}
        webhook_data = {
            "webhook_received": True,
            "orderId": order_id,
            "status": status,
            "rrn": rrn,
            "amount": str(amount) if amount else None
        }
        
        # Merge webhook data into existing api_response
        updated_api_response = {**current_api_response, **webhook_data}
        
        # If payment is successful and transaction was pending, schedule delayed credit
        if txn_status == "success" and payin_txn.status == "pending" and not payin_txn.refunded:
            # Schedule wallet credit after delay (stored in database, processed by scheduler)
            from datetime import datetime, timedelta
            from app.core.config import settings
            scheduled_time = datetime.utcnow() + timedelta(seconds=settings.PAYIN_CREDIT_DELAY_SECONDS)
            self.payin_repo.update(
                str(payin_txn.id),
                scheduled_credit_at=scheduled_time
            )
            logger.info(f"Scheduled wallet credit for txnid {payin_txn.txnid} at {scheduled_time} (delay: {settings.PAYIN_CREDIT_DELAY_SECONDS}s)")
        
        # Update transaction
        updated_payin = self.payin_repo.update(
            str(payin_txn.id),
            status=txn_status,
            rrn=rrn,
            api_response=updated_api_response
        )
        WebhookSenderService(db).send_webhook("payin", payin_txn.txnid)
        return updated_payin

