"""
Payout service with business logic
"""

import uuid
import secrets
import string
from typing import Optional, Dict, Any
from decimal import Decimal
from sqlalchemy.orm import Session
from app.repositories.payout_transaction_repository import PayoutTransactionRepository
from app.repositories.service_repository import ServiceRepository
from app.repositories.user_service_repository import UserServiceRepository
from app.repositories.user_repository import UserRepository
from app.services.wallet_service import WalletService
from app.services.providers.provider_factory import ProviderFactory
from app.core.exceptions import ValidationError, NotFoundError
from app.common.enums import UserServiceStatus, TransactionType
from app.models.payout_transaction import PayoutTransaction
from uuid import UUID


class PayoutService:
    """Payout service with business logic"""

    def __init__(self, db: Session):
        self.payout_repo = PayoutTransactionRepository(db)
        self.service_repo = ServiceRepository(db)
        self.user_service_repo = UserServiceRepository(db)
        self.user_repo = UserRepository(db)
        self.wallet_service = WalletService(db)
        self.db = db

    def check_service_status(self, user_id: UUID, service_code: str, db: Session) -> bool:
        """
        Check if user has active service for the given service code
        
        Args:
            user_id: User ID
            service_code: Service code (e.g., "payout")
            db: Database session
            
        Returns:
            True if user has active service, False otherwise
            
        Raises:
            NotFoundError: If service not found
            ValidationError: If service is not active for user or service itself is not active
        """
        from app.common.enums import ServiceStatus
        
        # Get service by code
        service = self.service_repo.get_by_code(service_code)
        if not service:
            raise NotFoundError(f"Service with code '{service_code}' not found")
        
        # Check if the service itself is active (in services table)
        if service.status != ServiceStatus.ACTIVE.value:
            raise ValidationError(f"Service '{service_code}' is not active. Service status: {service.status}")
        
        # Check user service status
        user_service = self.user_service_repo.get_by_user_and_service(str(user_id), str(service.id))
        if not user_service:
            raise ValidationError(f"Service '{service_code}' is not assigned to user")
        
        if user_service.status != UserServiceStatus.ACTIVE.value:
            raise ValidationError(f"Service '{service_code}' is not active for user")
        
        return True

    def initiate_payout(
        self,
        user_id: UUID,
        amount: Decimal,
        charge: Decimal,
        bene_name: str,
        bene_ifsc: str,
        bene_acc_no: str,
        user_reference_id: str,
        provider_name: str = None,
        db: Session = None
    ) -> PayoutTransaction:
        """
        Initiate a payout transaction
        
        Steps:
        1. Check service status for payout
        2. Debit wallet (amount + charge)
        3. Create payout transaction record
        4. Call rupeeflow API via provider
        5. Update transaction with API response
        
        Args:
            user_id: User ID
            amount: Payout amount
            charge: Service charge
            bene_name: Beneficiary name
            bene_ifsc: Beneficiary IFSC code
            bene_acc_no: Beneficiary account number
            user_reference_id: User-provided reference ID
            db: Database session
            
        Returns:
            Created PayoutTransaction instance
        """
        # Step 1: Check service status for payout
        self.check_service_status(user_id, "payout", db)
        
        # Step 2: Get provider (defaults to rupeeflow for now)
        # TODO: In future, implement user provider preference system
        provider_name = provider_name or "rupeeflow"
        
        # Step 3: Debit wallet (amount + charge)
        # Generate a single transaction ID that will be used for both wallet and payout transactions
        # This ensures both transactions share the same txnid for tracking purposes
        # Format: PYT + random alphanumeric string (12 characters)
        total_debit = amount + charge
        random_string = ''.join(secrets.choice(string.ascii_uppercase + string.digits) for _ in range(12))
        txnid = f"PYT{random_string}"
        
        # Create wallet transaction with the txnid
        wallet_txn = self.wallet_service.process_transaction(
            user_id=user_id,
            txn_type="payout",
            transaction_type=TransactionType.DEBIT.value,
            amount=total_debit,
            txnid=txnid,  # Same txnid used for tracking
            db=db,
            update_main=True
        )
        
        # Step 4: Generate provider_reference_id (apitxnid) before creating transaction
        # This ensures it's set immediately, preventing race conditions with webhooks
        # For Unitpay: generate apitxnid in format pay + 8 numeric digits
        # Use user_reference_id as apitxnid if it matches format (pay*)
        if provider_name == "unitpay":
            if user_reference_id and user_reference_id.startswith("pay") and len(user_reference_id) >= 4:
                provider_reference_id = user_reference_id
            else:
                # Generate apitxnid in format: pay + 8 numeric digits (e.g., pay5657878)
                random_suffix = ''.join(secrets.choice(string.digits) for _ in range(8))
                provider_reference_id = f"pay{random_suffix}"
        else:
            # For other providers, set to None initially, will be set from API response
            provider_reference_id = None
        
        # Step 5: Create payout transaction record with the same txnid
        # The payout txnid and wallet txnid are the same for tracking purposes
        # Check if user_reference_id already exists
        existing_txn = self.payout_repo.get_by_user_reference_id(user_reference_id)
        if existing_txn:
            raise ValidationError(f"Reference ID '{user_reference_id}' already exists. Please use a unique reference ID.")
        
        payout_txn = self.payout_repo.create(
            user_id=user_id,
            txnid=txnid,  # Same txnid as wallet transaction for tracking
            user_reference_id=user_reference_id,
            amount=amount,
            charge=charge,
            bene_name=bene_name,
            bene_ifsc=bene_ifsc,
            bene_acc_no=bene_acc_no,
            status="pending",
            api_provider=provider_name,
            provider_reference_id=provider_reference_id  # Set immediately for Unitpay
        )
        
        # Step 6: Get user's phone number for beneficiary mobile
        user = self.user_repo.get_by_id(str(user_id))
        beneficiary_mobile = user.phone if user and user.phone else None
        
        # Step 7: Get provider instance from factory
        provider = ProviderFactory.get_payout_provider(provider_name)
        # Pass provider_reference_id to provider so it uses the same one
        payout_response = provider.create_payout(
            txnid=txnid,
            amount=amount,
            bene_name=bene_name,
            bene_ifsc=bene_ifsc,
            bene_acc_no=bene_acc_no,
            user_reference_id=user_reference_id,
            beneficiary_mobile=beneficiary_mobile,
            provider_reference_id=provider_reference_id  # Pass the pre-generated apitxnid
        )
        
        # Step 8: Update transaction with API response
        # Convert PayoutResponse to dict for storage
        api_response = payout_response.to_dict()
        
        # Map API response status to transaction status
        api_status = payout_response.status
        if api_status == "success":
            txn_status = "success"
        elif api_status == "failed":
            txn_status = "failed"
        elif api_status == "error":
            txn_status = "error"
        else:
            txn_status = "pending"
        
        # Use provider_reference_id from response if not already set (for non-Unitpay providers)
        # For Unitpay, provider_reference_id is already set, but update if response has different value
        response_provider_reference_id = payout_response.reference_id
        if not provider_reference_id:
            provider_reference_id = response_provider_reference_id
        elif response_provider_reference_id and response_provider_reference_id != provider_reference_id:
            # If response has different reference_id, use it (shouldn't happen for Unitpay)
            provider_reference_id = response_provider_reference_id
        
        updated_payout = self.payout_repo.update(
            str(payout_txn.id),
            status=txn_status,
            api_response=api_response,
            provider_reference_id=provider_reference_id  # Update if changed, or keep existing
        )
        
        # Step 8: If API returns not success and not pending, refund the transaction
        if txn_status not in ["success", "pending"]:
            # Process refund for the wallet transaction
            try:
                self.wallet_service.process_refund(
                    user_id=user_id,
                    original_txnid=txnid,
                    txn_type="payout_refund",
                    amount=total_debit,  # Refund the total amount (amount + charge)
                    refund_txnid=txnid,  # Use same txnid for refund
                    db=db,
                    update_main=True
                )
                
                # Mark payout transaction as refunded
                updated_payout = self.payout_repo.update(
                    str(payout_txn.id),
                    refunded=True
                )
            except Exception as e:
                # Log the error but don't fail the transaction
                # The transaction status is already updated to failed/error
                # Refund can be processed manually if needed
                pass
        
        # Invalidate payout stats cache when transaction is created/updated
        try:
            from app.core.cache import get_cache
            cache = get_cache()
            cache.delete_pattern("payout_stats:")
        except Exception:
            # Don't fail if cache invalidation fails
            pass
        
        return updated_payout

    def get_payout_status(self, user_reference_id: str, user_id: UUID, db: Session) -> PayoutTransaction:
        """
        Get payout transaction status by user reference ID
        
        Args:
            user_reference_id: User-provided reference ID
            user_id: User ID
            db: Database session
            
        Returns:
            PayoutTransaction instance
            
        Raises:
            NotFoundError: If transaction not found
        """
        payout_txn = self.payout_repo.get_by_user_reference_id(user_reference_id)
        if not payout_txn:
            raise NotFoundError(f"Payout transaction with reference ID '{user_reference_id}' not found")
        
        # Verify transaction belongs to user
        if str(payout_txn.user_id) != str(user_id):
            raise NotFoundError(f"Payout transaction with reference ID '{user_reference_id}' not found")
        
        return payout_txn

    def get_transactions(
        self,
        user_id: Optional[UUID] = None,
        db: Session = None,
        skip: int = 0,
        limit: int = 100,
        username: Optional[str] = None,
        status: Optional[str] = None,
        txnid: Optional[str] = None,
        user_reference_id: Optional[str] = None
    ) -> list[PayoutTransaction]:
        """
        Get payout transaction history with optional filtering
        
        Args:
            user_id: User ID (optional, if None returns all transactions)
            db: Database session
            skip: Number of records to skip
            limit: Maximum number of records to return
            username: Username (email) to filter by (optional)
            status: Transaction status to filter by (optional)
            txnid: Transaction ID to filter by (optional)
            user_reference_id: User reference ID to filter by (optional)
            
        Returns:
            List of PayoutTransaction instances
        """
        from app.models.user import User
        
        # Build query with all filters applied BEFORE pagination
        query = self.db.query(self.payout_repo.model)
        
        # Apply user_id filter
        if user_id:
            query = query.filter(self.payout_repo.model.user_id == user_id)
        
        # Apply username filter (requires join)
        if username:
            query = query.join(User, self.payout_repo.model.user_id == User.id).filter(
                User.email.ilike(f"%{username}%")
            )
        
        # Apply status filter
        if status:
            query = query.filter(self.payout_repo.model.status.ilike(f"%{status}%"))
        
        # Apply txnid filter
        if txnid:
            query = query.filter(self.payout_repo.model.txnid == txnid)
        
        # Apply user_reference_id filter
        if user_reference_id:
            query = query.filter(self.payout_repo.model.user_reference_id == user_reference_id)
        
        # Apply ordering and pagination AFTER all filters
        transactions = query.order_by(
            self.payout_repo.model.created_at.desc()
        ).offset(skip).limit(limit).all()
        
        return transactions
    
    def handle_webhook(
        self,
        order_id: str,
        status: str,
        txn_ref_id: str = None,
        payout_id: str = None,
        amount: Decimal = None,
        db: Session = None
    ) -> PayoutTransaction:
        """
        Handle webhook callback from payment provider
        
        Args:
            order_id: Provider's order ID (stored as provider_reference_id)
            status: Status from provider (SUCCESS, FAILED, PENDING)
            txn_ref_id: Bank transaction reference ID (UTR) if available
            payout_id: Original payout ID (our txnid)
            amount: Transaction amount (for validation)
            db: Database session
            
        Returns:
            Updated PayoutTransaction instance
            
        Raises:
            NotFoundError: If transaction not found
        """
        # Find transaction by provider reference ID (orderId)
        # This is the primary lookup method - order_id should be the provider_reference_id
        import logging
        import time
        logger = logging.getLogger(__name__)
        logger.info(f"[Payout Webhook] Looking up transaction with order_id: '{order_id}'")
        
        # Try multiple lookup strategies with retries to handle race conditions
        # where webhook arrives before provider_reference_id is committed
        max_retries = 3
        retry_delay = 0.5  # seconds
        
        for attempt in range(max_retries):
            if attempt > 0:
                logger.info(f"[Payout Webhook] Retry attempt {attempt + 1}/{max_retries} after {retry_delay}s delay")
                time.sleep(retry_delay)
                # Refresh database session to see latest commits
                self.db.expire_all()
            
            # Strategy 1: Look up by provider_reference_id (primary method)
            payout_txn = self.payout_repo.get_by_provider_reference_id(order_id)
            if payout_txn:
                logger.info(f"[Payout Webhook] Found transaction by provider_reference_id: txnid={payout_txn.txnid}, status={payout_txn.status}")
                break
            
            # Strategy 2: Look up by apitxnid in api_response JSON (handles race condition)
            # This works even if provider_reference_id column isn't set yet
            if order_id.startswith('pay'):  # Unitpay apitxnid format
                payout_txn = self.payout_repo.get_by_apitxnid_in_response(order_id)
                if payout_txn:
                    logger.info(f"[Payout Webhook] Found transaction by apitxnid in api_response: txnid={payout_txn.txnid}, status={payout_txn.status}")
                    break
            
            # Strategy 3: Try looking up by txnid if provider_reference_id lookup fails
            if not payout_txn:
                payout_txn = self.payout_repo.get_by_txnid(order_id)
                if payout_txn:
                    logger.info(f"[Payout Webhook] Found transaction by txnid: txnid={payout_txn.txnid}, status={payout_txn.status}")
                    break
            
            # Strategy 4: If payout_id is provided, try looking up by txnid using payout_id
            # (payout_id from webhook might be our internal txnid)
            if not payout_txn and payout_id:
                payout_txn = self.payout_repo.get_by_txnid(payout_id)
                if payout_txn:
                    logger.info(f"[Payout Webhook] Found transaction by payout_id: txnid={payout_txn.txnid}, status={payout_txn.status}")
                    break
            
            if payout_txn:
                break
        
        if not payout_txn:
            # Log available fields for debugging
            logger.error(
                f"[Payout Webhook] Transaction not found - order_id: '{order_id}', "
                f"payout_id: '{payout_id}', txn_ref_id: '{txn_ref_id}'. "
                f"Attempted lookups: provider_reference_id='{order_id}', txnid='{order_id}'"
            )
            raise NotFoundError(f"Payout transaction with order ID '{order_id}' not found")
        # Map provider status to our status
        status_upper = status.upper()
        if status_upper == "SUCCESS":
            txn_status = "success"
        elif status_upper == "FAILED":
            txn_status = "failed"
        elif status_upper == "PENDING":
            txn_status = "pending"
        else:
            txn_status = "pending"
        
        # Update api_response to include webhook data
        current_api_response = payout_txn.api_response or {}
        webhook_data = {
            "webhook_received": True,
            "orderId": order_id,
            "status": status,
            "txnRefId": txn_ref_id,
            "payoutId": payout_id
        }
        
        # Merge webhook data into existing api_response
        updated_api_response = {**current_api_response, **webhook_data}
        
        # Update transaction
        updated_payout = self.payout_repo.update(
            str(payout_txn.id),
            rrn=txn_ref_id,
            status=txn_status,
            api_response=updated_api_response
        )
        
        # If webhook status is FAILED and transaction not already refunded, process refund
        if txn_status == "failed" and not payout_txn.refunded:
            try:
                # Calculate total amount to refund (amount + charge)
                total_refund = payout_txn.amount + payout_txn.charge
                
                # Process refund for the wallet transaction
                self.wallet_service.process_refund(
                    user_id=payout_txn.user_id,
                    original_txnid=payout_txn.txnid,
                    txn_type="payout_refund",
                    amount=total_refund,
                    refund_txnid=payout_txn.txnid,  # Use same txnid for refund
                    db=db,
                    update_main=True
                )
                
                # Mark payout transaction as refunded
                updated_payout = self.payout_repo.update(
                    str(payout_txn.id),
                    refunded=True
                )
            except Exception as e:
                # Log the error but don't fail the webhook processing
                # The transaction status is already updated to failed
                # Refund can be processed manually if needed
                pass
        
        # Invalidate payout stats cache when transaction status is updated via webhook
        try:
            from app.core.cache import get_cache
            cache = get_cache()
            cache.delete_pattern("payout_stats:")
        except Exception:
            # Don't fail if cache invalidation fails
            pass
        
        return updated_payout

