"""
Base provider interface for payin providers
"""

from abc import ABC, abstractmethod
from typing import Dict, Any
from decimal import Decimal


class PayinResponse:
    """Standardized payin response format"""
    
    def __init__(
        self,
        status: str,
        message: str,
        reference_id: str = None,
        txnid: str = None,
        amount: str = None,
        payment_url: str = None,
        qr_text: str = None,
        raw_response: Dict[str, Any] = None
    ):
        """
        Initialize payin response
        
        Args:
            status: Response status (success, failed, error, pending)
            message: Response message
            reference_id: Provider reference ID (orderId)
            txnid: Transaction ID
            amount: Transaction amount
            payment_url: Payment URL for user to complete payment
            qr_text: QR code text/intent for UPI payment
            raw_response: Raw API response for debugging
        """
        self.status = status
        self.message = message
        self.reference_id = reference_id
        self.txnid = txnid
        self.amount = amount
        self.payment_url = payment_url
        self.qr_text = qr_text
        self.raw_response = raw_response or {}
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert response to dictionary"""
        return {
            "status": self.status,
            "message": self.message,
            "reference_id": self.reference_id,
            "txnid": self.txnid,
            "amount": self.amount,
            "payment_url": self.payment_url,
            "qr_text": self.qr_text,
            "raw_response": self.raw_response
        }


class BasePayinProvider(ABC):
    """Base abstract class for payin providers"""
    
    @abstractmethod
    def create_payin(
        self,
        txnid: str,
        amount: Decimal,
        name: str,
        email: str,
        mobile: str,
        vpa: str = None,
        user_reference_id: str = None,
        **kwargs
    ) -> PayinResponse:
        """
        Create a payin transaction (payment link/QR)
        
        Args:
            txnid: Transaction ID
            amount: Payin amount
            name: Payer name
            email: Payer email
            mobile: Payer mobile number
            vpa: Payer VPA (Virtual Payment Address) - optional
            user_reference_id: User-provided reference ID
            **kwargs: Additional provider-specific parameters
            
        Returns:
            PayinResponse with standardized format
        """
        pass

