"""
Base provider interface for payout providers
"""

from abc import ABC, abstractmethod
from typing import Dict, Any
from decimal import Decimal


class PayoutResponse:
    """Standardized payout response format"""
    
    def __init__(
        self,
        status: str,
        message: str,
        reference_id: str = None,
        txnid: str = None,
        amount: str = None,
        raw_response: Dict[str, Any] = None
    ):
        """
        Initialize payout response
        
        Args:
            status: Response status (success, failed, error, pending)
            message: Response message
            reference_id: Provider reference ID
            txnid: Transaction ID
            amount: Transaction amount
            raw_response: Raw API response for debugging
        """
        self.status = status
        self.message = message
        self.reference_id = reference_id
        self.txnid = txnid
        self.amount = amount
        self.raw_response = raw_response or {}
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert response to dictionary"""
        return {
            "status": self.status,
            "message": self.message,
            "reference_id": self.reference_id,
            "txnid": self.txnid,
            "amount": self.amount,
            "raw_response": self.raw_response
        }


class BasePayoutProvider(ABC):
    """Base abstract class for payout providers"""
    
    @abstractmethod
    def create_payout(
        self,
        txnid: str,
        amount: Decimal,
        bene_name: str,
        bene_ifsc: str,
        bene_acc_no: str,
        user_reference_id: str,
        **kwargs
    ) -> PayoutResponse:
        """
        Create a payout transaction
        
        Args:
            txnid: Transaction ID
            amount: Payout amount
            bene_name: Beneficiary name
            bene_ifsc: Beneficiary IFSC code
            bene_acc_no: Beneficiary account number
            user_reference_id: User-provided reference ID
            **kwargs: Additional provider-specific parameters
            
        Returns:
            PayoutResponse with standardized format
        """
        pass

