"""
MytPiPay (myt) payin provider implementation.

API: https://banking.mytpipay.com/api/upi-collection/createorder
Request: form fields - api_token, amount, mobile, name, email, order_id (numeric)
Success response: status, data.order_id, data.gateway_order_id, data.amount, data.qr_string

order_id is generated as numeric (MYT API requires number). Stored in provider_reference_id
for webhook correlation; user_reference_id remains client-defined.
"""

import requests
import time
import secrets
from decimal import Decimal
from app.services.providers.base_payin_provider import BasePayinProvider, PayinResponse
from app.core.config import settings
from app.core.exceptions import ValidationError


class MytPayinProvider(BasePayinProvider):
    """MytPiPay (myt) payin provider - UPI collection create order"""

    def __init__(self):
        """Initialize MytPiPay payin provider"""
        self.api_token = settings.MYT_API_TOKEN
        self.api_url = settings.MYT_PAYIN_URL

        if not self.api_token:
            raise ValidationError("MytPiPay API token not configured")

    def create_payin(
        self,
        txnid: str,
        amount: Decimal,
        name: str,
        email: str,
        mobile: str,
        vpa: str = None,
        user_reference_id: str = None,
        **kwargs
    ) -> PayinResponse:
        """
        Create a payin (UPI collection order) via MytPiPay API.

        Sends form data: api_token, amount, mobile, name, email, order_id.
        order_id is generated as numeric (MYT requires number). Returned as reference_id
        and stored in provider_reference_id for webhook correlation.
        """
        # Generate numeric order_id, 10-20 digits (MYT API requirement); stored as provider_reference_id
        ts_ms = int(time.time() * 1000)  # 13 digits
        order_id = ts_ms * 1_000_000 + secrets.randbelow(1_000_000)  # 16-19 digits
        amount_str = str(int(amount)) if amount == int(amount) else str(amount)

        payload = {
            "api_token": self.api_token,
            "amount": amount_str,
            "mobile": str(mobile).strip(),
            "name": str(name).strip(),
            "email": str(email).strip(),
            "order_id": order_id,
        }

        try:
            # API expects form data (multipart or x-www-form-urlencoded)
            response = requests.post(
                self.api_url,
                data=payload,
                timeout=30,
            )
            response_data = response.json() if response.text else {}

            if response.status_code != 200:
                err_msg = (
                    response_data.get("error")
                    or response_data.get("message")
                    or f"API returned status {response.status_code}"
                )
                if isinstance(err_msg, dict):
                    err_msg = str(err_msg)
                return PayinResponse(
                    status="failed",
                    message=str(err_msg),
                    reference_id=None,
                    txnid=txnid,
                    amount=amount_str,
                    payment_url=None,
                    qr_text=None,
                    raw_response=response_data,
                )

            api_status = (response_data.get("status") or "").lower()
            data = response_data.get("data") or {}

            if api_status == "success" and data:
                # Use our numeric order_id (stored as provider_reference_id for webhook lookup)
                qr_string = data.get("qr_string") or ""
                return PayinResponse(
                    status="success",
                    message="Order created successfully",
                    reference_id=str(order_id),
                    txnid=txnid,
                    amount=amount_str,
                    payment_url=qr_string or None,
                    qr_text=qr_string,
                    raw_response=response_data,
                )

            err = response_data.get("error") or response_data.get("message") or "Payin request failed"
            if isinstance(err, dict):
                err = str(err)
            return PayinResponse(
                status="failed",
                message=str(err),
                reference_id=None,
                txnid=txnid,
                amount=amount_str,
                payment_url=None,
                qr_text=None,
                raw_response=response_data,
            )

        except requests.exceptions.RequestException as e:
            raw_response = {}
            err_msg = str(e)
            if hasattr(e, "response") and e.response is not None:
                try:
                    raw_response = e.response.json()
                    err_msg = raw_response.get("error", raw_response.get("message", err_msg))
                except Exception:
                    raw_response = {"status_code": getattr(e.response, "status_code", None), "error": err_msg}
            return PayinResponse(
                status="error",
                message=f"MytPiPay API error: {err_msg}",
                reference_id=None,
                txnid=txnid,
                amount=str(amount),
                payment_url=None,
                qr_text=None,
                raw_response=raw_response,
            )
