"""
Provider Factory for dynamic provider selection
"""

from typing import Optional, Dict, Type
from app.services.providers.base_provider import BasePayoutProvider
from app.services.providers.base_payin_provider import BasePayinProvider
from app.services.providers.rupeeflow_provider import RupeeflowProvider
from app.services.providers.rupeeflow_payin_provider import RupeeflowPayinProvider
from app.services.providers.unitpay_payin_provider import UnitpayPayinProvider
from app.services.providers.unitpay_payout_provider import UnitpayPayoutProvider
from app.services.providers.myt_payin_provider import MytPayinProvider
from app.core.exceptions import ValidationError


class ProviderFactory:
    """Factory class for creating payment providers dynamically"""
    
    # Registry of available payout providers
    _payout_providers: Dict[str, Type[BasePayoutProvider]] = {
        "rupeeflow": RupeeflowProvider,
        "unitpay": UnitpayPayoutProvider,
        # Add more providers here as they are implemented
        # "razorpay": RazorpayProvider,
        # "paytm": PaytmProvider,
    }
    
    # Registry of available payin providers
    _payin_providers: Dict[str, Type[BasePayinProvider]] = {
        "rupeeflow": RupeeflowPayinProvider,
        "unitpay": UnitpayPayinProvider,
        "myt": MytPayinProvider,
    }
    
    @classmethod
    def get_payout_provider(cls, provider_name: Optional[str] = None) -> BasePayoutProvider:
        """
        Get payout provider instance
        
        Args:
            provider_name: Name of the provider (e.g., "rupeeflow")
                          If None, returns default provider
            
        Returns:
            BasePayoutProvider instance
            
        Raises:
            ValidationError: If provider not found
        """
        # Default to rupeeflow if not specified
        provider_name = provider_name or "rupeeflow"
        provider_name = provider_name.lower()
        
        if provider_name not in cls._payout_providers:
            available = ", ".join(cls._payout_providers.keys())
            raise ValidationError(
                f"Payout provider '{provider_name}' not found. "
                f"Available providers: {available}"
            )
        
        provider_class = cls._payout_providers[provider_name]
        return provider_class()
    
    @classmethod
    def get_payin_provider(cls, provider_name: Optional[str] = None) -> BasePayinProvider:
        """
        Get payin provider instance
        
        Args:
            provider_name: Name of the provider (e.g., "rupeeflow")
                          If None, returns default provider
            
        Returns:
            BasePayinProvider instance
            
        Raises:
            ValidationError: If provider not found
        """
        # Default to rupeeflow if not specified
        provider_name = provider_name or "rupeeflow"
        provider_name = provider_name.lower()
        
        if provider_name not in cls._payin_providers:
            available = ", ".join(cls._payin_providers.keys())
            raise ValidationError(
                f"Payin provider '{provider_name}' not found. "
                f"Available providers: {available}"
            )
        
        provider_class = cls._payin_providers[provider_name]
        return provider_class()
    
    @classmethod
    def register_payout_provider(cls, name: str, provider_class: Type[BasePayoutProvider]):
        """
        Register a new payout provider (useful for dynamic registration)
        
        Args:
            name: Provider name
            provider_class: Provider class that extends BasePayoutProvider
        """
        cls._payout_providers[name.lower()] = provider_class
    
    @classmethod
    def register_payin_provider(cls, name: str, provider_class: Type[BasePayinProvider]):
        """
        Register a new payin provider (useful for dynamic registration)
        
        Args:
            name: Provider name
            provider_class: Provider class that extends BasePayinProvider
        """
        cls._payin_providers[name.lower()] = provider_class
    
    @classmethod
    def list_payout_providers(cls) -> list[str]:
        """Get list of available payout provider names"""
        return list(cls._payout_providers.keys())
    
    @classmethod
    def list_payin_providers(cls) -> list[str]:
        """Get list of available payin provider names"""
        return list(cls._payin_providers.keys())

