"""
Rupeeflow payin provider implementation
"""

import base64
import requests
from typing import Dict, Any
from decimal import Decimal
from app.services.providers.base_payin_provider import BasePayinProvider, PayinResponse
from app.core.config import settings
from app.core.exceptions import ValidationError


class RupeeflowPayinProvider(BasePayinProvider):
    """Rupeeflow payin provider"""
    
    def __init__(self):
        """Initialize Rupeeflow payin provider"""
        self.api_key = settings.RUPEEFLOW_API_KEY
        self.secret_key = settings.RUPEEFLOW_SECRET_KEY
        self.api_url = settings.RUPEEFLOW_PAYIN_URL
        
        if not self.api_key or not self.secret_key:
            raise ValidationError("Rupeeflow API credentials not configured")
    
    def _get_auth_headers(self) -> Dict[str, str]:
        """Get authorization headers for API requests"""
        credentials = f"{self.api_key}:{self.secret_key}"
        encoded_credentials = base64.b64encode(credentials.encode()).decode()
        
        return {
            "Content-Type": "application/json",
            "Authorization": f"Basic {encoded_credentials}"
        }
    
    def create_payin(
        self,
        txnid: str,
        amount: Decimal,
        name: str,
        email: str,
        mobile: str,
        vpa: str = None,
        user_reference_id: str = None,
        **kwargs
    ) -> PayinResponse:
        """
        Create a payin via Rupeeflow API
        
        Args:
            txnid: Transaction ID
            amount: Payin amount
            name: Payer name
            email: Payer email
            mobile: Payer mobile number
            vpa: Payer VPA (Virtual Payment Address) - optional
            user_reference_id: User-provided reference ID (used as orderId)
            **kwargs: Additional parameters
            
        Returns:
            PayinResponse with standardized format
        """
        # Use user_reference_id as orderId if provided, otherwise use txnid
        order_id = user_reference_id or txnid
        
        # Prepare request payload - match exact Postman format
        # Convert Decimal amount to appropriate numeric type
        # API expects integer for whole numbers, float for decimals
        amount_float = float(amount)
        if amount_float.is_integer():
            amount_value = int(amount_float)
        else:
            amount_value = amount_float
        
        # Ensure all string fields are stripped of whitespace
        payload = {
            "orderId": str(order_id).strip(),
            "amount": amount_value,
            "name": str(name).strip(),
            "email": str(email).strip(),
            "mobile": str(mobile).strip()
        }
        
        # Add VPA if provided (optional field)
        if vpa:
            payload["vpa"] = str(vpa).strip()
        
        # Prepare headers
        headers = self._get_auth_headers()
        
        # Make API call
        try:
            response = requests.post(
                self.api_url,
                json=payload,
                headers=headers,
                timeout=30
            )
            
            # Parse response - don't raise for status yet, check response first
            response_data = response.json()
            
            # Check if request failed
            if response.status_code != 200 and response.status_code != 201:
                error_message = response_data.get("message", f"API returned status {response.status_code}")
                return PayinResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=str(amount),
                    payment_url=None,
                    qr_text=None,
                    raw_response=response_data
                )
            
            # Raise for status only if we haven't handled it above
            response.raise_for_status()
            
            # Map rupeeflow response to standardized format
            # API returns statusCode 201 on success with success: true
            if response_data.get("success") and response.status_code in [200, 201]:
                # Extract data from response
                data = response_data.get("data", {})
                order_id_response = data.get("orderId", order_id)
                intent = data.get("intent", "")  # UPI payment intent/QR text
                message = data.get("message") or response_data.get("message", "Payment Link Generated successfully")
                
                # The intent is the UPI payment link/QR text
                qr_text = intent
                payment_url = intent  # Can be used as payment URL for UPI
                
                return PayinResponse(
                    status="success",
                    message=message,
                    reference_id=order_id_response,
                    txnid=txnid,
                    amount=str(amount),
                    payment_url=payment_url,
                    qr_text=qr_text,
                    raw_response=response_data
                )
            else:
                error_message = response_data.get("message", "Payin API call failed")
                return PayinResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=str(amount),
                    payment_url=None,
                    qr_text=None,
                    raw_response=response_data
                )
                
        except requests.exceptions.RequestException as e:
            # Handle network errors, timeouts, etc.
            error_msg = str(e)
            raw_response = {}
            
            if hasattr(e, 'response') and e.response is not None:
                try:
                    error_data = e.response.json()
                    error_msg = error_data.get("message", error_msg)
                    raw_response = error_data
                except:
                    error_msg = f"HTTP {e.response.status_code}: {error_msg}"
                    raw_response = {"status_code": e.response.status_code, "error": error_msg}
            
            return PayinResponse(
                status="error",
                message=f"Rupeeflow API error: {error_msg}",
                reference_id=None,
                txnid=txnid,
                amount=str(amount),
                payment_url=None,
                qr_text=None,
                raw_response=raw_response
            )

