"""
Rupeeflow payout provider implementation
"""

import base64
import requests
from typing import Dict, Any
from decimal import Decimal
from app.services.providers.base_provider import BasePayoutProvider, PayoutResponse
from app.core.config import settings
from app.core.exceptions import ValidationError
import logging
import json

logger = logging.getLogger(__name__)
class RupeeflowProvider(BasePayoutProvider):
    """Rupeeflow payout provider"""
    
    def __init__(self):
        """Initialize Rupeeflow provider"""
        self.api_key = settings.RUPEEFLOW_API_KEY
        self.secret_key = settings.RUPEEFLOW_SECRET_KEY
        self.api_url = settings.RUPEEFLOW_API_URL
        self.status_url = settings.RUPEEFLOW_STATUS_URL
        
        if not self.api_key or not self.secret_key:
            raise ValidationError("Rupeeflow API credentials not configured")
    
    def _get_auth_headers(self) -> Dict[str, str]:
        """Get authorization headers for API requests"""
        credentials = f"{self.api_key}:{self.secret_key}"
        encoded_credentials = base64.b64encode(credentials.encode()).decode()
        
        return {
            "Content-Type": "application/json",
            "Authorization": f"Basic {encoded_credentials}"
        }
    
    def create_payout(
        self,
        txnid: str,
        amount: Decimal,
        bene_name: str,
        bene_ifsc: str,
        bene_acc_no: str,
        user_reference_id: str,
        beneficiary_mobile: str = None,
        **kwargs
    ) -> PayoutResponse:
        """
        Create a payout via Rupeeflow API
        
        Args:
            txnid: Transaction ID
            amount: Payout amount
            bene_name: Beneficiary name
            bene_ifsc: Beneficiary IFSC code
            bene_acc_no: Beneficiary account number
            user_reference_id: User-provided reference ID
            **kwargs: Additional parameters (purpose, payment_mode, remarks, bank_name)
            
        Returns:
            PayoutResponse with standardized format
        """
        # Extract bank name from IFSC (first 4 characters typically represent the bank)
        # IFSC format: XXXX0XXXXX where first 4 chars are bank code
        bank_name = kwargs.get("bank_name")
        if not bank_name:
            bank_name = bene_ifsc[:4].upper() if len(bene_ifsc) >= 4 else "UNKNOWN"
        
        # Prepare request payload
        payout_data = {
            "amount": float(amount),
            "purpose": kwargs.get("purpose", "Payout"),
            "beneficiaryName": bene_name,
            "bankName": bank_name,
            "accountNumber": bene_acc_no,
            "ifscCode": bene_ifsc.upper(),
            "remarks": kwargs.get("remarks", f"Payout for transaction {txnid}"),
            "transferMode": kwargs.get("payment_mode", "IMPS"),
            "payoutId": txnid
        }
        
        # Add beneficiary mobile if provided
        if beneficiary_mobile:
            payout_data["beneficiaryMobile"] = beneficiary_mobile
        
        payload = {
            "data": [payout_data]
        }
        
        # Prepare headers
        headers = self._get_auth_headers()
        
        # Make API call
        try:
            response = requests.post(
                self.api_url,
                json=payload,
                headers=headers,
                timeout=30
            )
            
            # Parse response
            response.raise_for_status()  # Raises an HTTPError for bad responses
            response_data = response.json()
            logger.info("-------------------------------- Payout Payload") 
            logger.info(payload)
            logger.info("-------------------------------- Payout Payload")
            # Map rupeeflow response to standardized format
            # API returns statusCode 201 on success with success: true
            if response_data.get("success") and response.status_code in [200, 201]:
                # Extract orderId from response (this is the provider reference ID)
                # Response structure: data.payoutOrders[0].orderId
                reference_id = None
                payout_status = "pending"
                
                if "data" in response_data:
                    data = response_data["data"]
                    
                    # Extract orderId from payoutOrders array
                    if "payoutOrders" in data and len(data["payoutOrders"]) > 0:
                        payout_order = data["payoutOrders"][0]
                        reference_id = payout_order.get("orderId")
                        
                        # Map rupeeflow status to our status
                        rf_status = payout_order.get("status", "PENDING").upper()
                        if rf_status == "SUCCESS":
                            payout_status = "success"
                        elif rf_status == "FAILED":
                            payout_status = "failed"
                        elif rf_status == "PENDING":
                            payout_status = "pending"
                        else:
                            payout_status = "pending"
                    
                    # Get message from data.message or top-level message
                    message = data.get("message") or response_data.get("message", "Payout initiated successfully")
                else:
                    message = response_data.get("message", "Payout initiated successfully")
                
                return PayoutResponse(
                    status=payout_status,
                    message=message,
                    reference_id=reference_id or txnid,
                    txnid=txnid,
                    amount=str(amount),
                    raw_response=response_data
                )
            else:
                error_message = response_data.get("message", "Payout API call failed")
                return PayoutResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=str(amount),
                    raw_response=response_data
                )
                
        except requests.exceptions.RequestException as e:
            # Handle network errors, timeouts, etc.
            error_msg = str(e)
            raw_response = {}
            
            if hasattr(e, 'response') and e.response is not None:
                try:
                    error_data = e.response.json()
                    error_msg = error_data.get("message", error_msg)
                    raw_response = error_data
                except:
                    error_msg = f"HTTP {e.response.status_code}: {error_msg}"
                    raw_response = {"status_code": e.response.status_code, "error": error_msg}
            
            return PayoutResponse(
                status="error",
                message=f"Rupeeflow API error: {error_msg}",
                reference_id=None,
                txnid=txnid,
                amount=str(amount),
                raw_response=raw_response
            )
    
    def check_payout_status(self, order_id: str) -> Dict[str, Any]:
        """
        Check payout transaction status using order ID
        
        Args:
            order_id: Rupeeflow order ID (from create_payout response)
            
        Returns:
            Dictionary with status information:
            - status: SUCCESS, PENDING, or FAILED
            - txnRefId: Bank transaction reference number (UTR) if available
            - payoutId: Original payout ID
            - raw_response: Full API response
        """
        payload = {
            "orderId": order_id
        }
        
        headers = self._get_auth_headers()
        
        try:
            response = requests.post(
                self.status_url,
                json=payload,
                headers=headers,
                timeout=30
            )
            
            response.raise_for_status()
            response_data = response.json()
            
            if response_data.get("success") and "data" in response_data:
                data = response_data["data"]
                return {
                    "status": data.get("status", "PENDING"),
                    "txnRefId": data.get("txnRefId"),
                    "payoutId": data.get("payoutId"),
                    "orderId": data.get("orderId"),
                    "raw_response": response_data
                }
            else:
                return {
                    "status": "UNKNOWN",
                    "txnRefId": None,
                    "payoutId": None,
                    "orderId": order_id,
                    "raw_response": response_data
                }
                
        except requests.exceptions.RequestException as e:
            error_msg = str(e)
            if hasattr(e, 'response') and e.response is not None:
                try:
                    error_data = e.response.json()
                    error_msg = error_data.get("message", error_msg)
                except:
                    error_msg = f"HTTP {e.response.status_code}: {error_msg}"
            
            return {
                "status": "ERROR",
                "txnRefId": None,
                "payoutId": None,
                "orderId": order_id,
                "error": error_msg,
                "raw_response": {}
            }

