"""
Unitpay payin provider implementation
"""

import requests
from typing import Dict, Any
from decimal import Decimal
from app.services.providers.base_payin_provider import BasePayinProvider, PayinResponse
from app.core.config import settings
from app.core.exceptions import ValidationError


class UnitpayPayinProvider(BasePayinProvider):
    """Unitpay payin provider"""
    
    def __init__(self):
        """Initialize Unitpay payin provider"""
        self.token = settings.UNITPAY_TOKEN
        self.api_url = settings.UNITPAY_API_URL
        self.webhook_base_url = settings.UNITPAY_WEBHOOK_BASE_URL
        
        if not self.token:
            raise ValidationError("Unitpay API token not configured")
    
    def create_payin(
        self,
        txnid: str,
        amount: Decimal,
        name: str,
        email: str,
        mobile: str,
        vpa: str = None,
        user_reference_id: str = None,
        callback_url: str = None,
        **kwargs
    ) -> PayinResponse:
        """
        Create a payin via Unitpay API
        
        Args:
            txnid: Transaction ID
            amount: Payin amount
            name: Payer name
            email: Payer email
            mobile: Payer mobile number
            vpa: Payer VPA (Virtual Payment Address) - optional
            user_reference_id: User-provided reference ID (used as txnid)
            callback_url: Webhook callback URL (optional, will be constructed if not provided)
            **kwargs: Additional parameters
            
        Returns:
            PayinResponse with standardized format
        """
        # Use user_reference_id as txnid if provided, otherwise use txnid
        api_txnid = user_reference_id or txnid
        
        # Construct callback URL if not provided
        if not callback_url:
            base_url = self.webhook_base_url.rstrip('/')
            callback_url = f"{base_url}/webhooks/unitpay/payin"
        
        # Prepare request payload
        # Convert Decimal amount to string
        amount_str = str(amount)
        
        payload = {
            "token": self.token,
            "type": "dynamic",
            "amount": amount_str,
            "email": str(email).strip(),
            "mobile": str(mobile).strip(),
            "name": str(name).strip(),
            "txnid": api_txnid,
            "callback": callback_url
        }
        
        # Prepare headers
        headers = {
            "Content-Type": "application/json"
        }
        
        # Make API call
        try:
            response = requests.post(
                self.api_url,
                json=payload,
                headers=headers,
                timeout=30
            )
            
            # Parse response
            response_data = response.json()
            
            # Check if request failed
            if response.status_code != 200:
                error_message = response_data.get("message", f"API returned status {response.status_code}")
                return PayinResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=amount_str,
                    payment_url=None,
                    qr_text=None,
                    raw_response=response_data
                )
            
            # Check response statuscode
            statuscode = response_data.get("statuscode", "").upper()
            
            if statuscode == "TXN":
                # Transaction successful - extract payment details
                payment_url = response_data.get("payment_url", "")
                upi_string = response_data.get("upi_string", "")
                orderid = response_data.get("orderid", response_data.get("txnid", api_txnid))
                message = response_data.get("message", "Transaction Successfull")
                
                # Use payment_url or upi_string as QR text
                qr_text = payment_url or upi_string
                
                return PayinResponse(
                    status="success",
                    message=message,
                    reference_id=orderid,
                    txnid=txnid,
                    amount=amount_str,
                    payment_url=payment_url or upi_string,
                    qr_text=qr_text,
                    raw_response=response_data
                )
            else:
                error_message = response_data.get("message", "Payin API call failed")
                return PayinResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=amount_str,
                    payment_url=None,
                    qr_text=None,
                    raw_response=response_data
                )
                
        except requests.exceptions.RequestException as e:
            # Handle network errors, timeouts, etc.
            error_msg = str(e)
            raw_response = {}
            
            if hasattr(e, 'response') and e.response is not None:
                try:
                    error_data = e.response.json()
                    error_msg = error_data.get("message", error_msg)
                    raw_response = error_data
                except:
                    error_msg = f"HTTP {e.response.status_code}: {error_msg}"
                    raw_response = {"status_code": e.response.status_code, "error": error_msg}
            
            return PayinResponse(
                status="error",
                message=f"Unitpay API error: {error_msg}",
                reference_id=None,
                txnid=txnid,
                amount=str(amount),
                payment_url=None,
                qr_text=None,
                raw_response=raw_response
            )

