"""
Unitpay payout provider implementation
"""

import requests
import secrets
import string
from typing import Dict, Any
from decimal import Decimal
from app.services.providers.base_provider import BasePayoutProvider, PayoutResponse
from app.core.config import settings
from app.core.exceptions import ValidationError


class UnitpayPayoutProvider(BasePayoutProvider):
    """Unitpay payout provider"""
    
    def __init__(self):
        """Initialize Unitpay payout provider"""
        self.token = settings.UNITPAY_TOKEN
        self.api_url = settings.UNITPAY_PAYOUT_URL
        self.webhook_base_url = settings.UNITPAY_WEBHOOK_BASE_URL
        
        if not self.token:
            raise ValidationError("Unitpay API token not configured")
    
    def create_payout(
        self,
        txnid: str,
        amount: Decimal,
        bene_name: str,
        bene_ifsc: str,
        bene_acc_no: str,
        user_reference_id: str,
        beneficiary_mobile: str = None,
        callback_url: str = None,
        **kwargs
    ) -> PayoutResponse:
        """
        Create a payout via Unitpay API
        
        Args:
            txnid: Transaction ID
            amount: Payout amount
            bene_name: Beneficiary name
            bene_ifsc: Beneficiary IFSC code
            bene_acc_no: Beneficiary account number
            user_reference_id: User-provided reference ID (used as apitxnid)
            beneficiary_mobile: Beneficiary mobile number
            callback_url: Webhook callback URL (optional, will be constructed if not provided)
            **kwargs: Additional parameters (bank, mode)
            
        Returns:
            PayoutResponse with standardized format
        """
        # Use provider_reference_id from kwargs if provided (pre-generated by service)
        # Otherwise use user_reference_id as apitxnid if it matches format (pay*)
        # Otherwise generate apitxnid in format: pay + 8 numeric digits
        api_txnid = kwargs.get("provider_reference_id")
        if not api_txnid:
            if user_reference_id and user_reference_id.startswith("pay") and len(user_reference_id) >= 4:
                api_txnid = user_reference_id
            else:
                # Generate apitxnid in format: pay + 8 numeric digits (e.g., pay5657878)
                random_suffix = ''.join(secrets.choice(string.digits) for _ in range(8))
                api_txnid = f"pay{random_suffix}"
        
        # Construct callback URL if not provided
        if not callback_url:
            base_url = self.webhook_base_url.rstrip('/')
            callback_url = f"{base_url}/webhooks/unitpay/payout"
        
        # Get bank name from kwargs or extract from IFSC
        bank = kwargs.get("bank", "Bank")
        if bank == "Bank":
            # Try to extract bank name from IFSC (first 4 characters)
            if len(bene_ifsc) >= 4:
                bank = bene_ifsc[:4].upper()
        
        # Get payment mode from kwargs (default: IMPS)
        mode = kwargs.get("mode", "IMPS")
        
        # Prepare request payload
        amount_str = str(amount)
        
        payload = {
            "token": self.token,
            "mobile": beneficiary_mobile or "",
            "account": bene_acc_no,
            "bank": bank,
            "mode": mode,
            "ifsc": bene_ifsc.upper(),
            "name": bene_name,
            "amount": amount_str,
            "apitxnid": api_txnid,
            "callback": callback_url
        }
        
        # Prepare headers
        headers = {
            "Content-Type": "application/json"
        }
        
        # Make API call
        try:
            response = requests.post(
                self.api_url,
                json=payload,
                headers=headers,
                timeout=30
            )
            
            # Parse response
            response_data = response.json()
            
            # Check if request failed
            if response.status_code != 200:
                error_message = response_data.get("message", f"API returned status {response.status_code}")
                return PayoutResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=amount_str,
                    raw_response=response_data
                )
            
            # Check response statuscode
            statuscode = response_data.get("statuscode", "").upper()
            
            # NOTE:
            # For Unitpay payout, statuscode == "TXN" does NOT mean final success.
            # It only means the payout request is accepted and is now pending.
            # Final success / failure (and UTR etc.) will be received via webhook.
            if statuscode == "TXN":
                # Treat as pending; store provider reference so webhook can match it
                apitxnid = response_data.get("apitxnid", api_txnid)
                message = response_data.get("message", "Transaction Pending")
                
                return PayoutResponse(
                    status="pending",
                    message=message,
                    reference_id=apitxnid,  # Use apitxnid as provider_reference_id
                    txnid=txnid,
                    amount=amount_str,
                    raw_response=response_data
                )
            else:
                error_message = response_data.get("message", "Payout API call failed")
                return PayoutResponse(
                    status="failed",
                    message=error_message,
                    reference_id=None,
                    txnid=txnid,
                    amount=amount_str,
                    raw_response=response_data
                )
                
        except requests.exceptions.RequestException as e:
            # Handle network errors, timeouts, etc.
            error_msg = str(e)
            raw_response = {}
            
            if hasattr(e, 'response') and e.response is not None:
                try:
                    error_data = e.response.json()
                    error_msg = error_data.get("message", error_msg)
                    raw_response = error_data
                except:
                    error_msg = f"HTTP {e.response.status_code}: {error_msg}"
                    raw_response = {"status_code": e.response.status_code, "error": error_msg}
            
            return PayoutResponse(
                status="error",
                message=f"Unitpay API error: {error_msg}",
                reference_id=None,
                txnid=txnid,
                amount=str(amount),
                raw_response=raw_response
            )

