"""
Service service with business logic
"""

from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.service_repository import ServiceRepository
from app.schemas.service import ServiceCreate, ServiceUpdate
from app.core.exceptions import ValidationError, ConflictError
from app.common.enums import ServiceStatus
from app.models.service import Service


class ServiceService(BaseService[Service, ServiceCreate, ServiceUpdate]):
    """Service service with business logic"""

    def __init__(self, db: Session):
        repository = ServiceRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: ServiceCreate, db: Session) -> dict:
        """Validate service creation"""
        validated_data = data.model_dump()
        
        # Check if code already exists
        service_repo = ServiceRepository(db)
        existing_service = service_repo.get_by_code(data.code)
        if existing_service:
            raise ConflictError("Service code already exists")
        
        # Set default status
        validated_data["status"] = ServiceStatus.ACTIVE.value
        return validated_data

    def validate_update(self, data: ServiceUpdate, db: Session) -> dict:
        """Validate service update"""
        validated_data = data.model_dump(exclude_unset=True)
        
        # Validate status if being updated
        if "status" in validated_data:
            new_status = validated_data["status"]
            if new_status not in [s.value for s in ServiceStatus]:
                raise ValidationError(f"Invalid status: {new_status}")
        
        # Check code uniqueness if code is being updated
        if "code" in validated_data:
            service_repo = ServiceRepository(db)
            existing_service = service_repo.get_by_code(validated_data["code"])
            if existing_service and existing_service.id != self.repository.model.id:
                raise ConflictError("Service code already exists")
        
        return validated_data

