"""
User service with business logic
"""

from typing import Optional
from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.user_repository import UserRepository
from app.schemas.user import UserCreate, UserUpdate, UserResponse
from app.core.security import get_password_hash, verify_password, create_access_token
from app.core.exceptions import ValidationError, ConflictError, UnauthorizedError
from app.models.user import User
from datetime import timedelta


class UserService(BaseService[User, UserCreate, UserUpdate]):
    """User service with business logic"""

    def __init__(self, db: Session):
        repository = UserRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: UserCreate, db: Session) -> dict:
        """Validate user creation"""
        # Check if email already exists
        user_repo = UserRepository(db)
        existing_user = user_repo.get_by_email(data.email)
        if existing_user:
            raise ConflictError("Email already registered")
        
        validated_data = data.model_dump(exclude={"password"})
        validated_data["password_hash"] = get_password_hash(data.password)
        return validated_data

    def register(self, data: UserCreate) -> tuple[User, str]:
        """
        Register a new user and return user with access token
        
        Returns:
            Tuple of (User, access_token)
        """
        user_data = self.validate_create(data, self.db)
        user = self.repository.create(**user_data)
        
        # Create access token
        access_token = create_access_token(
            data={"sub": str(user.id), "role": "user"}
        )
        
        return user, access_token

    def login(self, email: str, password: str) -> tuple[User, str]:
        """
        Authenticate user and return user with access token
        
        Returns:
            Tuple of (User, access_token)
            
        Raises:
            UnauthorizedError: If credentials are invalid
        """
        user_repo = UserRepository(self.db)
        user = user_repo.get_by_email(email)
        
        if not user or not verify_password(password, user.password_hash):
            raise UnauthorizedError("Invalid email or password")
        
        if not user.is_active:
            raise UnauthorizedError("User account is inactive")
        
        # Create access token
        access_token = create_access_token(
            data={"sub": str(user.id), "role": "user"}
        )
        
        return user, access_token

