"""
User Service service with business logic
"""

from sqlalchemy.orm import Session
from app.common.base_service import BaseService
from app.repositories.user_service_repository import UserServiceRepository
from app.repositories.service_repository import ServiceRepository
from app.repositories.user_repository import UserRepository
from app.schemas.user_service import UserServiceCreate, UserServiceUpdate
from app.core.exceptions import ValidationError, NotFoundError, ConflictError
from app.common.enums import UserServiceStatus
from app.models.user_service import UserService
from uuid import UUID


class UserServiceService(BaseService[UserService, UserServiceCreate, UserServiceUpdate]):
    """User Service service with business logic"""

    def __init__(self, db: Session):
        repository = UserServiceRepository(db)
        super().__init__(repository)
        self.db = db

    def validate_create(self, data: UserServiceCreate, db: Session) -> dict:
        """Validate user service creation"""
        validated_data = data.model_dump()
        # Set default status
        validated_data["status"] = UserServiceStatus.PENDING.value
        return validated_data

    def validate_update(self, data: UserServiceUpdate, db: Session) -> dict:
        """Validate user service update"""
        validated_data = data.model_dump(exclude_unset=True)
        
        # Validate status if being updated
        if "status" in validated_data:
            new_status = validated_data["status"]
            if new_status not in [s.value for s in UserServiceStatus]:
                raise ValidationError(f"Invalid status: {new_status}")
        
        return validated_data

    def create(self, data: UserServiceCreate, db: Session, user_id: UUID) -> UserService:
        """
        Create user service assignment
        
        Args:
            data: User Service creation data
            db: Database session
            user_id: User ID
            
        Returns:
            Created UserService instance
        """
        # Verify service exists
        service_repo = ServiceRepository(db)
        service = service_repo.get_by_id(str(data.service_id))
        if not service:
            raise NotFoundError("Service")
        
        # Check if user-service relationship already exists
        user_service_repo = UserServiceRepository(db)
        existing = user_service_repo.get_by_user_and_service(str(user_id), str(data.service_id))
        if existing:
            raise ConflictError("User already has this service assigned")
        
        validated_data = self.validate_create(data, db)
        validated_data["user_id"] = user_id
        return self.repository.create(**validated_data)

    def get_user_services_with_details(self, user_id: UUID, db: Session) -> list:
        """
        Get all services for a user with their status
        Returns all services, marking which ones the user has as active
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            List of services with user service status
        """
        from app.common.enums import UserServiceStatus
        
        # Get all services
        service_repo = ServiceRepository(db)
        all_services = service_repo.get_all()
        
        # Get user's services
        user_service_repo = UserServiceRepository(db)
        user_services = user_service_repo.get_by_user_id(str(user_id))
        
        # Create a map of service_id -> user_service
        user_service_map = {}
        for us in user_services:
            user_service_map[str(us.service_id)] = us
        
        # Build response
        result = []
        for service in all_services:
            user_service = user_service_map.get(str(service.id))
            
            # Determine if service is active for this user
            is_active = False
            if user_service:
                is_active = user_service.status == UserServiceStatus.ACTIVE.value
            
            result.append({
                "service_id": service.id,
                "service_name": service.name,
                "service_code": service.code,
                "service_description": service.description,
                "service_price": str(service.price),
                "service_currency": service.currency,
                "service_status": service.status,
                "user_service_id": user_service.id if user_service else None,
                "user_service_status": user_service.status if user_service else None,
                "user_service_notes": user_service.notes if user_service else None,
                "is_active": is_active,
                "created_at": user_service.created_at if user_service else None,
                "updated_at": user_service.updated_at if user_service else None,
            })
        
        return result

    def activate_service_for_user(self, user_id: UUID, service_id: UUID, db: Session, notes: str = None) -> UserService:
        """
        Activate a service for a user
        Creates user_service record if it doesn't exist, otherwise updates status to active
        
        Args:
            user_id: User ID
            service_id: Service ID
            db: Database session
            notes: Optional notes
            
        Returns:
            UserService instance
        """
        from app.common.enums import UserServiceStatus
        
        # Verify service exists
        service_repo = ServiceRepository(db)
        service = service_repo.get_by_id(str(service_id))
        if not service:
            raise NotFoundError("Service")
        
        # Check if user-service relationship already exists
        user_service_repo = UserServiceRepository(db)
        existing = user_service_repo.get_by_user_and_service(str(user_id), str(service_id))
        
        if existing:
            # Update existing record to active
            return self.repository.update(str(existing.id), status=UserServiceStatus.ACTIVE.value)
        else:
            # Create new record with active status
            validated_data = {
                "user_id": user_id,
                "service_id": service_id,
                "status": UserServiceStatus.ACTIVE.value,
                "notes": notes
            }
            return self.repository.create(**validated_data)

    def deactivate_service_for_user(self, user_id: UUID, service_id: UUID, db: Session) -> UserService:
        """
        Deactivate a service for a user
        
        Args:
            user_id: User ID
            service_id: Service ID
            db: Database session
            
        Returns:
            UserService instance
        """
        from app.common.enums import UserServiceStatus
        
        # Check if user-service relationship exists
        user_service_repo = UserServiceRepository(db)
        existing = user_service_repo.get_by_user_and_service(str(user_id), str(service_id))
        
        if not existing:
            raise NotFoundError("User service not found")
        
        # Update status to inactive
        return self.repository.update(str(existing.id), status=UserServiceStatus.INACTIVE.value)

