"""
Wallet service with business logic
"""

import uuid
from typing import Optional
from decimal import Decimal
from sqlalchemy.orm import Session
from app.repositories.wallet_repository import WalletRepository
from app.repositories.wallet_transaction_repository import WalletTransactionRepository
from app.schemas.wallet import WalletCreate, WalletUpdate
from app.schemas.wallet_transaction import WalletTransactionCreate
from app.core.exceptions import ValidationError, NotFoundError
from app.common.enums import WalletStatus, TransactionType
from app.models.wallet import Wallet
from app.models.wallet_transaction import WalletTransaction
from uuid import UUID


class WalletService:
    """Wallet service with business logic"""

    def __init__(self, db: Session):
        self.wallet_repo = WalletRepository(db)
        self.transaction_repo = WalletTransactionRepository(db)
        self.db = db

    def get_or_create_wallet(self, user_id: UUID, db: Session, for_update: bool = False) -> Wallet:
        """
        Get wallet for user, create if doesn't exist
        
        Args:
            user_id: User ID
            db: Database session
            for_update: If True, lock the row for update (SELECT FOR UPDATE)
                       This prevents race conditions in concurrent transactions
            
        Returns:
            Wallet instance
        """
        wallet = self.wallet_repo.get_by_user_id(str(user_id), for_update=for_update)
        if not wallet:
            # Create new wallet with zero balance
            # Handle race condition: if another transaction creates it concurrently,
            # we'll get a unique constraint violation, then retry with lock
            try:
                wallet = self.wallet_repo.create(
                    user_id=user_id,
                    main=Decimal('0.00'),
                    reserve=Decimal('0.00'),
                    status=WalletStatus.ACTIVE.value
                )
            except Exception as e:
                # If wallet was created by another transaction (unique constraint violation),
                # get it again with lock if requested
                from sqlalchemy.exc import IntegrityError
                if isinstance(e, IntegrityError) or "unique constraint" in str(e).lower():
                    if for_update:
                        wallet = self.wallet_repo.get_by_user_id(str(user_id), for_update=True)
                    else:
                        wallet = self.wallet_repo.get_by_user_id(str(user_id))
                    if not wallet:
                        raise  # Re-raise if still not found
                else:
                    raise  # Re-raise other errors
        return wallet

    def get_wallet(self, user_id: UUID, db: Session) -> Wallet:
        """
        Get wallet for user
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            Wallet instance
        """
        wallet = self.wallet_repo.get_by_user_id(str(user_id))
        if not wallet:
            raise NotFoundError("Wallet not found")
        return wallet

    def update_status(self, user_id: UUID, status: str, db: Session) -> Wallet:
        """
        Update wallet status
        
        Args:
            user_id: User ID
            status: New status
            db: Database session
            
        Returns:
            Updated Wallet instance
        """
        wallet = self.get_wallet(user_id, db)
        if status not in [s.value for s in WalletStatus]:
            raise ValidationError(f"Invalid wallet status: {status}")
        return self.wallet_repo.update(str(wallet.id), status=status)

    def process_transaction(
        self,
        user_id: UUID,
        txn_type: str,
        transaction_type: str,
        amount: Decimal,
        txnid: str,
        db: Session,
        update_main: bool = True
    ) -> WalletTransaction:
        """
        Process a wallet transaction (debit or credit)
        
        Business Rule: If wallet status is not active, only credit entries are allowed
        
        Args:
            user_id: User ID
            txn_type: Reason code for transaction
            transaction_type: 'debit' or 'credit'
            amount: Transaction amount
            txnid: Transaction ID (can be reused for refunds)
            db: Database session
            update_main: If True, update main balance; if False, update reserve balance
            
        Returns:
            Created WalletTransaction instance
            
        Raises:
            ValidationError: If transaction is invalid
        """
        # Validate transaction type
        if transaction_type not in [TransactionType.DEBIT.value, TransactionType.CREDIT.value]:
            raise ValidationError(f"Invalid transaction type: {transaction_type}. Must be 'debit' or 'credit'")

        # Get or create wallet WITH ROW LOCK to prevent race conditions
        # This ensures that concurrent transactions wait for each other
        wallet = self.get_or_create_wallet(user_id, db, for_update=True)

        # Business Rule: If wallet is not active, only credit entries are allowed
        if wallet.status != WalletStatus.ACTIVE.value and transaction_type == TransactionType.DEBIT.value:
            raise ValidationError(
                f"Cannot process debit transaction. Wallet status is '{wallet.status}'. "
                "Only credit entries are allowed for inactive wallets."
            )

        # Get current balance (now safely locked)
        if update_main:
            opening_balance = wallet.main
        else:
            opening_balance = wallet.reserve

        # Calculate closing balance
        if transaction_type == TransactionType.CREDIT.value:
            closing_balance = opening_balance + amount
        else:  # DEBIT
            # Check sufficient balance
            if opening_balance < amount:
                raise ValidationError(f"Insufficient balance. Available: {opening_balance}, Required: {amount}")
            closing_balance = opening_balance - amount

        # Update wallet balance (still within the locked transaction)
        if update_main:
            self.wallet_repo.update(str(wallet.id), main=closing_balance)
        else:
            self.wallet_repo.update(str(wallet.id), reserve=closing_balance)

        # Refresh wallet to get updated balance
        wallet = self.wallet_repo.get_by_id(str(wallet.id))

        # Create transaction log
        transaction = self.transaction_repo.create(
            user_id=user_id,
            wallet_id=wallet.id,
            txn_type=txn_type,
            type=transaction_type,
            amount=amount,
            opening=opening_balance,
            closing=closing_balance,
            txnid=txnid,
            refunded=False,
            is_refund=False
        )

        return transaction

    def process_refund(
        self,
        user_id: UUID,
        original_txnid: str,
        txn_type: str,
        amount: Decimal,
        refund_txnid: str,
        db: Session,
        update_main: bool = True
    ) -> WalletTransaction:
        """
        Process a refund transaction
        
        Marks the original transaction as refunded and creates a new credit entry with is_refund=true
        
        Args:
            user_id: User ID
            original_txnid: Transaction ID of the original transaction to refund
            txn_type: Reason code for refund transaction
            amount: Refund amount
            refund_txnid: Transaction ID for the refund (can be same as original_txnid)
            db: Database session
            update_main: If True, update main balance; if False, update reserve balance
            
        Returns:
            Created refund WalletTransaction instance
            
        Raises:
            NotFoundError: If original transaction not found
            ValidationError: If original transaction already refunded or invalid
        """
        # Find the original transaction (get the most recent one if multiple exist)
        # Filter for non-refunded, non-refund transactions
        from app.models.wallet_transaction import WalletTransaction
        original_txn = self.db.query(WalletTransaction).filter(
            WalletTransaction.txnid == original_txnid,
            WalletTransaction.user_id == user_id,
            WalletTransaction.is_refund == False,
            WalletTransaction.refunded == False
        ).order_by(WalletTransaction.created_at.desc()).first()
        
        if not original_txn:
            # Check if transaction exists but is already refunded
            existing_txn = self.db.query(WalletTransaction).filter(
                WalletTransaction.txnid == original_txnid,
                WalletTransaction.user_id == user_id,
                WalletTransaction.is_refund == False
            ).first()
            
            if existing_txn and existing_txn.refunded:
                raise ValidationError(f"Transaction with ID '{original_txnid}' has already been refunded")
            else:
                raise NotFoundError(f"Original transaction with ID '{original_txnid}' not found")
        
        # Verify the original transaction is a debit
        if original_txn.type != TransactionType.DEBIT.value:
            raise ValidationError("Can only refund debit transactions")
        
        # Mark original transaction as refunded
        self.transaction_repo.update(str(original_txn.id), refunded=True)
        
        # Process refund as credit transaction
        refund_transaction = self.process_transaction(
            user_id=user_id,
            txn_type=txn_type,
            transaction_type=TransactionType.CREDIT.value,
            amount=amount,
            txnid=refund_txnid,
            db=db,
            update_main=update_main
        )
        
        # Mark the refund transaction as is_refund=True
        self.transaction_repo.update(str(refund_transaction.id), is_refund=True)
        
        # Refresh to get updated values
        return self.transaction_repo.get_by_id(str(refund_transaction.id))

    def get_transactions(
        self,
        user_id: Optional[UUID] = None,
        db: Session = None,
        skip: int = 0,
        limit: int = 100,
        username: Optional[str] = None,
        type: Optional[str] = None,
        txn_type: Optional[str] = None,
        refunded: Optional[bool] = None,
        is_refund: Optional[bool] = None,
        txnid: Optional[str] = None
    ) -> list[WalletTransaction]:
        """
        Get transaction history with optional filtering
        
        Args:
            user_id: User ID (optional, if None returns all transactions)
            db: Database session
            skip: Number of records to skip
            limit: Maximum number of records to return
            username: Username (email) to filter by (optional)
            type: Transaction type (debit/credit) to filter by (optional)
            txn_type: Transaction type/reason code to filter by (optional)
            refunded: Filter by refunded status (optional)
            is_refund: Filter by is_refund status (optional)
            txnid: Transaction ID to filter by (optional)
            
        Returns:
            List of WalletTransaction instances
        """
        from app.models.user import User
        
        # Build query with all filters applied BEFORE pagination
        query = self.db.query(self.transaction_repo.model)
        
        # Apply user_id filter
        if user_id:
            query = query.filter(self.transaction_repo.model.user_id == user_id)
        
        # Apply username filter (requires join)
        if username:
            query = query.join(User, self.transaction_repo.model.user_id == User.id).filter(
                User.email.ilike(f"%{username}%")
            )
        
        # Apply type filter
        if type:
            query = query.filter(self.transaction_repo.model.type == type)
        
        # Apply txn_type filter
        if txn_type:
            query = query.filter(self.transaction_repo.model.txn_type == txn_type)
        
        # Apply refunded filter
        if refunded is not None:
            query = query.filter(self.transaction_repo.model.refunded == refunded)
        
        # Apply is_refund filter
        if is_refund is not None:
            query = query.filter(self.transaction_repo.model.is_refund == is_refund)
        
        # Apply txnid filter
        if txnid:
            query = query.filter(self.transaction_repo.model.txnid == txnid)
        
        # Apply ordering and pagination AFTER all filters
        transactions = query.order_by(
            self.transaction_repo.model.created_at.desc()
        ).offset(skip).limit(limit).all()
        
        return transactions

