"""
Webhook Log service with business logic
"""

from typing import Optional, List, Dict, Any
from sqlalchemy.orm import Session
from app.repositories.webhook_log_repository import WebhookLogRepository
from app.models.webhook_log import WebhookLog
from uuid import UUID


class WebhookLogService:
    """Webhook Log service with business logic"""

    def __init__(self, db: Session):
        self.repository = WebhookLogRepository(db)
        self.db = db

    def create_log(
        self,
        user_id: UUID,
        service_id: UUID,
        txnid: str,
        payload: Dict[str, Any],
        response: Optional[Dict[str, Any]] = None,
        db: Session = None
    ) -> WebhookLog:
        """
        Create a webhook log entry
        
        Args:
            user_id: User ID
            service_id: Service ID
            txnid: Transaction ID
            payload: Webhook payload
            response: Webhook response (optional)
            db: Database session (optional, uses self.db if not provided)
            
        Returns:
            WebhookLog instance
        """
        if db is None:
            db = self.db
        
        return self.repository.create(
            user_id=user_id,
            service_id=service_id,
            txnid=txnid,
            payload=payload,
            response=response
        )

    def get_logs(
        self,
        skip: int = 0,
        limit: int = 100,
        user_id: Optional[UUID] = None,
        service_id: Optional[UUID] = None,
        txnid: Optional[str] = None,
        db: Session = None
    ) -> List[WebhookLog]:
        """
        Get webhook logs with optional filters
        
        Args:
            skip: Number of records to skip
            limit: Maximum number of records to return
            user_id: Filter by user ID (optional)
            service_id: Filter by service ID (optional)
            txnid: Filter by transaction ID (optional)
            db: Database session (optional)
            
        Returns:
            List of WebhookLog instances
        """
        if db is None:
            db = self.db
        
        user_id_str = str(user_id) if user_id else None
        service_id_str = str(service_id) if service_id else None
        return self.repository.get_all_with_filters(
            skip=skip,
            limit=limit,
            user_id=user_id_str,
            service_id=service_id_str,
            txnid=txnid
        )

    def count_logs(
        self,
        user_id: Optional[UUID] = None,
        service_id: Optional[UUID] = None,
        txnid: Optional[str] = None,
        db: Session = None
    ) -> int:
        """
        Count webhook logs with optional filters
        
        Args:
            user_id: Filter by user ID (optional)
            service_id: Filter by service ID (optional)
            txnid: Filter by transaction ID (optional)
            db: Database session (optional)
            
        Returns:
            Count of webhook logs
        """
        if db is None:
            db = self.db
        
        user_id_str = str(user_id) if user_id else None
        service_id_str = str(service_id) if service_id else None
        return self.repository.count_with_filters(
            user_id=user_id_str,
            service_id=service_id_str,
            txnid=txnid
        )

