"""
Webhook Sender Service
Handles sending webhooks to user's configured webhook URLs
"""

import httpx
import json
import logging
from typing import Optional, Dict, Any
from sqlalchemy.orm import Session
from uuid import UUID
from decimal import Decimal
from datetime import datetime

from app.repositories.service_repository import ServiceRepository
from app.repositories.webhook_url_repository import WebhookURLRepository
from app.repositories.api_key_repository import APIKeyRepository
from app.repositories.payout_transaction_repository import PayoutTransactionRepository
from app.repositories.payin_transaction_repository import PayinTransactionRepository
from app.repositories.wallet_transaction_repository import WalletTransactionRepository
from app.services.webhook_log_service import WebhookLogService
from app.core.exceptions import NotFoundError, ValidationError
from app.core.security import sign_webhook_payload, decrypt_api_secret

logger = logging.getLogger(__name__)


class WebhookSenderService:
    """Service for sending webhooks to user's configured URLs"""
    
    # HTTP client timeout settings
    TIMEOUT = 10.0  # 10 seconds timeout for webhook requests
    
    def __init__(self, db: Session):
        self.db = db
        self.service_repo = ServiceRepository(db)
        self.webhook_url_repo = WebhookURLRepository(db)
        self.api_key_repo = APIKeyRepository(db)
        self.payout_repo = PayoutTransactionRepository(db)
        self.payin_repo = PayinTransactionRepository(db)
        self.wallet_repo = WalletTransactionRepository(db)
        self.webhook_log_service = WebhookLogService(db)
    
    def send_webhook(
        self,
        service_code: str,
        txnid: str,
        payload: Optional[Dict[str, Any]] = None,
        db: Session = None
    ) -> Dict[str, Any]:
        """
        Send webhook to user's configured webhook URL
        
        Args:
            service_code: Service code (e.g., 'payout', 'payin', 'wallet')
            txnid: Transaction ID
            payload: Optional payload. If not provided, will be built from transaction data
            db: Database session (optional, uses self.db if not provided)
            
        Returns:
            Dictionary with webhook sending result:
            {
                'success': bool,
                'webhook_url': str,
                'response': dict or None,
                'error': str or None
            }
            
        Raises:
            NotFoundError: If service, transaction, or webhook URL not found
            ValidationError: If service_code or txnid is invalid
        """
        if db is None:
            db = self.db
        
        # Step 1: Get service by code
        service = self.service_repo.get_by_code(service_code)
        if not service:
            raise NotFoundError(f"Service with code '{service_code}' not found")
        
        service_id = service.id
        
        # Step 2: Get or build payload
        if payload is None:
            # Fetch transaction and build payload
            transaction_data = self._fetch_transaction_data(service_code, txnid, db)
            if not transaction_data:
                raise NotFoundError(f"Transaction with txnid '{txnid}' not found for service '{service_code}'")
            
            user_id = transaction_data['user_id']
            payload = self._build_payload_from_transaction(service_code, transaction_data)
        else:
            # If payload is provided, we need to get user_id from the transaction
            # We still need to fetch transaction to get user_id
            transaction_data = self._fetch_transaction_data(service_code, txnid, db)
            if not transaction_data:
                raise NotFoundError(f"Transaction with txnid '{txnid}' not found for service '{service_code}'")
            user_id = transaction_data['user_id']
        
        # Step 3: Get user's webhook URL
        webhook_url_obj = self.webhook_url_repo.get_by_user_id(str(user_id))
        if not webhook_url_obj:
            # User hasn't configured webhook URL, silently skip
            return {
                'success': False,
                'webhook_url': None,
                'response': None,
                'error': 'User has not configured webhook URL'
            }
        
        webhook_url = webhook_url_obj.url
        
        # Step 4: Get API secret for webhook signing
        api_key_obj = self.api_key_repo.get_by_user_id(str(user_id))
        if not api_key_obj or not api_key_obj.api_secret:
            # User doesn't have API key or secret, cannot sign webhook
            return {
                'success': False,
                'webhook_url': webhook_url,
                'response': None,
                'error': 'User does not have API key configured. Please generate API key first.'
            }
        
        # Decrypt API secret for signing
        try:
            api_secret = decrypt_api_secret(api_key_obj.api_secret)
            # Extract values for signature string
            user_reference_id = payload.get('user_reference_id', '')
            data = payload.get('data', {})
            status = data.get('status', '') if isinstance(data, dict) else ''
            signature_string = f"{user_reference_id}|{txnid}|{status}"
            # Log the decrypted secret and signature string used for webhook signing
            payload_json = json.dumps(payload, indent=2)
            logger.info(
                f"Webhook signing - User ID: {user_id}, Service: {service_code}, "
                f"Transaction ID: {txnid}, Decrypted API Secret: {api_secret}, "
                f"Signature String: {signature_string}, Payload: {payload_json}"
            )
        except Exception as e:
            return {
                'success': False,
                'webhook_url': webhook_url,
                'response': None,
                'error': f'Failed to decrypt API secret: {str(e)}'
            }
        
        # Step 5: Generate webhook signature using API secret
        signature = sign_webhook_payload(payload, api_secret)
        
        # Step 6: Send webhook HTTP request with signature
        try:
            response_data = self._send_http_request(webhook_url, payload, signature)
            
            # Step 7: Log webhook
            self.webhook_log_service.create_log(
                user_id=user_id,
                service_id=service_id,
                txnid=txnid,
                payload=payload,
                response=response_data.get('response'),
                db=db
            )
            
            return {
                'success': response_data['success'],
                'webhook_url': webhook_url,
                'response': response_data.get('response'),
                'error': response_data.get('error')
            }
            
        except Exception as e:
            # Log the error even if webhook sending failed
            error_response = {
                'error': str(e),
                'status_code': None
            }
            
            try:
                self.webhook_log_service.create_log(
                    user_id=user_id,
                    service_id=service_id,
                    txnid=txnid,
                    payload=payload,
                    response=error_response,
                    db=db
                )
            except Exception as log_error:
                # If logging fails, we can't do much, but don't fail the main operation
                pass
            
            return {
                'success': False,
                'webhook_url': webhook_url,
                'response': error_response,
                'error': str(e)
            }
    
    def _fetch_transaction_data(self, service_code: str, txnid: str, db: Session) -> Optional[Dict[str, Any]]:
        """
        Fetch transaction data based on service_code and txnid
        
        Args:
            service_code: Service code
            txnid: Transaction ID
            db: Database session
            
        Returns:
            Dictionary with transaction data or None if not found
        """
        transaction = None
        
        if service_code.lower() == 'payout':
            transaction = self.payout_repo.get_by_txnid(txnid)
            if transaction:
                return {
                    'user_id': transaction.user_id,
                    'txnid': transaction.txnid,
                    'user_reference_id': transaction.user_reference_id,
                    'amount': float(transaction.amount),
                    'charge': float(transaction.charge),
                    'bene_name': transaction.bene_name,
                    'bene_ifsc': transaction.bene_ifsc,
                    'bene_acc_no': transaction.bene_acc_no,
                    'status': transaction.status,
                    'rrn': transaction.rrn,
                    'api_provider': transaction.api_provider,
                    'provider_reference_id': transaction.provider_reference_id,
                    'refunded': transaction.refunded,
                    'created_at': transaction.created_at.isoformat() if transaction.created_at else None,
                    'updated_at': transaction.updated_at.isoformat() if transaction.updated_at else None,
                }
        
        elif service_code.lower() == 'payin':
            transaction = self.payin_repo.get_by_txnid(txnid)
            if transaction:
                return {
                    'user_id': transaction.user_id,
                    'txnid': transaction.txnid,
                    'user_reference_id': transaction.user_reference_id,
                    'amount': float(transaction.amount),
                    'charge': float(transaction.charge),
                    'payee_vpa': transaction.payee_vpa,
                    'payee_name': transaction.payee_name,
                    'status': transaction.status,
                    'rrn': transaction.rrn,
                    'api_provider': transaction.api_provider,
                    'provider_reference_id': transaction.provider_reference_id,
                    'qr_text': transaction.qr_text,
                    'payment_url': transaction.payment_url,
                    'refunded': transaction.refunded,
                    'created_at': transaction.created_at.isoformat() if transaction.created_at else None,
                    'updated_at': transaction.updated_at.isoformat() if transaction.updated_at else None,
                }
        
        elif service_code.lower() == 'wallet':
            transaction = self.wallet_repo.get_by_txnid(txnid)
            if transaction:
                return {
                    'user_id': transaction.user_id,
                    'wallet_id': str(transaction.wallet_id),
                    'txnid': transaction.txnid,
                    'txn_type': transaction.txn_type,
                    'type': transaction.type,
                    'amount': float(transaction.amount),
                    'opening': float(transaction.opening),
                    'closing': float(transaction.closing),
                    'refunded': transaction.refunded,
                    'is_refund': transaction.is_refund,
                    'created_at': transaction.created_at.isoformat() if transaction.created_at else None,
                    'updated_at': transaction.updated_at.isoformat() if transaction.updated_at else None,
                }
        
        # If service_code doesn't match known types, try to find in any transaction table
        # This allows flexibility for future service types
        if not transaction:
            # Try payout
            transaction = self.payout_repo.get_by_txnid(txnid)
            if transaction:
                return self._fetch_transaction_data('payout', txnid, db)
            
            # Try payin
            transaction = self.payin_repo.get_by_txnid(txnid)
            if transaction:
                return self._fetch_transaction_data('payin', txnid, db)
            
            # Try wallet
            transaction = self.wallet_repo.get_by_txnid(txnid)
            if transaction:
                return self._fetch_transaction_data('wallet', txnid, db)
        
        return None
    
    def _build_payload_from_transaction(self, service_code: str, transaction_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Build webhook payload from transaction data
        Format: {service: "payin", user_reference_id: "xxx", data: {}}
        Only includes user-relevant data, excludes internal fields like api_provider
        
        Args:
            service_code: Service code
            transaction_data: Transaction data dictionary
            
        Returns:
            Formatted payload dictionary
        """
        service = service_code.lower()
        
        if service == 'payout':
            payload = {
                'service': 'payout',
                'user_reference_id': transaction_data.get('user_reference_id'),
                'data': {
                    'txnid': transaction_data.get('txnid'),
                    'amount': transaction_data.get('amount'),
                    'charge': transaction_data.get('charge'),
                    'bene_name': transaction_data.get('bene_name'),
                    'bene_ifsc': transaction_data.get('bene_ifsc'),
                    'bene_acc_no': transaction_data.get('bene_acc_no'),
                    'status': transaction_data.get('status'),
                    'refunded': transaction_data.get('refunded'),
                    'rrn': transaction_data.get('rrn'),
                    'created_at': transaction_data.get('created_at'),
                    'updated_at': transaction_data.get('updated_at'),
                }
            }
        
        elif service == 'payin':
            payload = {
                'service': 'payin',
                'user_reference_id': transaction_data.get('user_reference_id'),
                'data': {
                    'txnid': transaction_data.get('txnid'),
                    'amount': transaction_data.get('amount'),
                    'charge': transaction_data.get('charge'),
                    'payee_vpa': transaction_data.get('payee_vpa'),
                    'payee_name': transaction_data.get('payee_name'),
                    'status': transaction_data.get('status'),
                    'qr_text': transaction_data.get('qr_text'),
                    'payment_url': transaction_data.get('payment_url'),
                    'refunded': transaction_data.get('refunded'),
                    'rrn': transaction_data.get('rrn'),
                    'created_at': transaction_data.get('created_at'),
                    'updated_at': transaction_data.get('updated_at'),
                }
            }
        
        elif service == 'wallet':
            # Wallet transactions don't have user_reference_id, use txnid as identifier
            payload = {
                'service': 'wallet',
                'user_reference_id': transaction_data.get('txnid'),  # Use txnid as reference for wallet
                'data': {
                    'txnid': transaction_data.get('txnid'),
                    'txn_type': transaction_data.get('txn_type'),
                    'type': transaction_data.get('type'),
                    'amount': transaction_data.get('amount'),
                    'opening': transaction_data.get('opening'),
                    'closing': transaction_data.get('closing'),
                    'refunded': transaction_data.get('refunded'),
                    'is_refund': transaction_data.get('is_refund'),
                    'created_at': transaction_data.get('created_at'),
                    'updated_at': transaction_data.get('updated_at'),
                }
            }
        
        else:
            # For unknown service codes, return minimal payload
            payload = {
                'service': service,
                'user_reference_id': transaction_data.get('user_reference_id') or transaction_data.get('txnid'),
                'data': {
                    k: v for k, v in transaction_data.items() 
                    if k not in ['user_id', 'api_provider', 'provider_reference_id']
                }
            }
        
        return payload
    
    def _send_http_request(self, webhook_url: str, payload: Dict[str, Any], signature: str) -> Dict[str, Any]:
        """
        Send HTTP POST request to webhook URL with signature
        
        Args:
            webhook_url: Webhook URL to send request to
            payload: Payload to send
            signature: HMAC-SHA256 signature of the payload
            
        Returns:
            Dictionary with response data:
            {
                'success': bool,
                'response': dict or None,
                'error': str or None
            }
        """
        try:
            with httpx.Client(timeout=self.TIMEOUT) as client:
                response = client.post(
                    webhook_url,
                    json=payload,
                    headers={
                        'Content-Type': 'application/json',
                        'User-Agent': 'Fintech-Webhook-Sender/1.0',
                        'X-Webhook-Signature': signature
                    }
                )
                
                # Try to parse JSON response
                try:
                    response_data = response.json()
                except (json.JSONDecodeError, ValueError):
                    response_data = {
                        'status_code': response.status_code,
                        'text': response.text[:500]  # Limit text length
                    }
                
                return {
                    'success': response.is_success,
                    'response': {
                        'status_code': response.status_code,
                        'data': response_data,
                        'headers': dict(response.headers)
                    },
                    'error': None if response.is_success else f"HTTP {response.status_code}"
                }
        
        except httpx.TimeoutException:
            return {
                'success': False,
                'response': {
                    'status_code': None,
                    'error': 'timeout',
                    'message': f'Request timeout after {self.TIMEOUT} seconds'
                },
                'error': 'Request timeout'
            }
        
        except httpx.RequestError as e:
            return {
                'success': False,
                'response': {
                    'status_code': None,
                    'error': 'request_error',
                    'message': str(e)
                },
                'error': str(e)
            }
        
        except Exception as e:
            return {
                'success': False,
                'response': {
                    'status_code': None,
                    'error': 'unknown_error',
                    'message': str(e)
                },
                'error': str(e)
            }

