"""
Webhook URL service with business logic
"""

from typing import Optional
from sqlalchemy.orm import Session
from app.repositories.webhook_url_repository import WebhookURLRepository
from app.schemas.webhook_url import WebhookURLCreate, WebhookURLUpdate
from app.core.exceptions import ValidationError
from app.models.webhook_url import WebhookURL
from uuid import UUID


class WebhookURLService:
    """Webhook URL service with business logic"""

    def __init__(self, db: Session):
        self.repository = WebhookURLRepository(db)
        self.db = db

    def set_webhook_url(self, user_id: UUID, url: str, db: Session) -> WebhookURL:
        """
        Set or update webhook URL for a user
        If user already has a webhook URL, updates it. Otherwise creates a new one.
        
        Args:
            user_id: User ID
            url: Webhook URL
            db: Database session
            
        Returns:
            WebhookURL instance
        """
        # Validate URL format
        if not url or len(url) > 500:
            raise ValidationError("Invalid webhook URL")
        
        # Check if user already has a webhook URL
        existing = self.repository.get_by_user_id(str(user_id))
        
        if existing:
            # Update existing webhook URL
            return self.repository.update(str(existing.id), url=url)
        else:
            # Create new webhook URL
            return self.repository.create(user_id=user_id, url=url)

    def get_by_user_id(self, user_id: UUID, db: Session) -> Optional[WebhookURL]:
        """
        Get webhook URL for a user
        
        Args:
            user_id: User ID
            db: Database session
            
        Returns:
            WebhookURL instance or None
        """
        return self.repository.get_by_user_id(str(user_id))

